package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginExecution;
import org.apache.maven.plugin.BuildPluginManager;
import org.apache.maven.plugin.InvalidPluginDescriptorException;
import org.apache.maven.plugin.MojoExecution;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugin.MojoNotFoundException;
import org.apache.maven.plugin.PluginConfigurationException;
import org.apache.maven.plugin.PluginDescriptorParsingException;
import org.apache.maven.plugin.PluginManagerException;
import org.apache.maven.plugin.PluginNotFoundException;
import org.apache.maven.plugin.PluginResolutionException;
import org.apache.maven.plugin.descriptor.MojoDescriptor;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.shared.invoker.CommandLineConfigurationException;
import org.apache.maven.shared.invoker.DefaultInvocationRequest;
import org.apache.maven.shared.invoker.InvocationRequest;
import org.apache.maven.shared.invoker.MavenCommandLineBuilder;
import org.apache.maven.shared.invoker.SystemOutHandler;
import org.apache.maven.shared.utils.cli.Commandline;
import org.apache.maven.shared.utils.cli.StreamPumper;
import org.apache.maven.shared.utils.logging.MessageUtils;
import org.codehaus.plexus.configuration.PlexusConfiguration;
import org.codehaus.plexus.util.cli.StreamFeeder;
import org.codehaus.plexus.util.xml.Xpp3Dom;
import org.codehaus.plexus.util.xml.Xpp3DomUtils;

import com.sap.cds.maven.plugin.AbstractCdsMojo;

/**
 * An abstract base class for all Mojos using the Maven invoker or Mojo executor.
 */
abstract class AbstractInvokerMojo extends AbstractCdsMojo {

	/*
	 * Details about supported default properties: \
	 * https://maven.apache.org/components/ref/3-LATEST/maven-model-builder/ \
	 * https://github.com/cko/predefined_maven_properties/blob/master/README.md
	 */

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${project.basedir}/pom.xml", readonly = true, required = true)
	private File pom;

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${maven.home}", readonly = true, required = true)
	private File mavenHome;

	/**
	 * The Maven BuildPluginManager component.
	 */
	@Component
	private BuildPluginManager pluginManager;

	protected InvocationRequest createRequest(List<String> goals) {
		InvocationRequest request = new DefaultInvocationRequest();
		request.setPomFile(this.pom);
		request.setGoals(goals);

		// forward active profile IDs to new maven instance
		List<String> activeProfileIds = super.session.getProjectBuildingRequest().getActiveProfileIds();
		request.setProfiles(activeProfileIds);

		// debug mode is enable with cmd line option -X
		request.setDebug(getLog().isDebugEnabled());

		// enable to batch mode to avoid warnings about missing input stream
		request.setBatchMode(true);

		// forward cmd line arguments like -Dcds.generate.codeOutputDirectory=... to invoked goals
		request.setProperties(super.session.getUserProperties());

		return request;
	}

	protected void executeMojo(Plugin plugin, String goal, PluginExecution pluginExec) throws MojoExecutionException {
		// get mojo descriptor for given plugin and goal
		MojoDescriptor mojoDescriptor = getMojoDescriptor(plugin, goal);

		executeMojo(mojoDescriptor, pluginExec);
	}

	protected Process executeMaven(InvocationRequest request) throws MojoExecutionException {
		// create a maven command line for given invocation request
		Commandline mvnCmdLine = createMvnCmdLine(request);

		return executeMaven(mvnCmdLine);
	}

	/**
	 * Creates a new maven command line for given invocation request.
	 *
	 * @param request the {@link InvocationRequest invocation request}, may not <code>null</code>.
	 * @return the newly created {@link Commandline command line} to execute maven in a new process.
	 * @throws MojoExecutionException if creation of command line failed
	 */
	private Commandline createMvnCmdLine(InvocationRequest request) throws MojoExecutionException {
		MavenCommandLineBuilder builder = new MavenCommandLineBuilder();
		builder.setMavenHome(this.mavenHome);
		try {
			return builder.build(request);
		} catch (CommandLineConfigurationException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	@SuppressWarnings("resource")
	private Process executeMaven(Commandline mvnCmdLine) throws MojoExecutionException {
		// remove quotes again
		String executable = unquote(mvnCmdLine.getExecutable());

		List<String> commandline = new ArrayList<>();
		commandline.add(executable);
		Collections.addAll(commandline, mvnCmdLine.getArguments());

		ProcessBuilder procBuilder = new ProcessBuilder(commandline);
		procBuilder.directory(mvnCmdLine.getWorkingDirectory());

		logInfo("Executing Maven command %s in working directory %s", procBuilder.command(), procBuilder.directory());

		Process mvnProcess;
		try {
			mvnProcess = procBuilder.start();

			StreamPumper pumper = new StreamPumper(mvnProcess.getInputStream(), new SystemOutHandler());
			pumper.start();
			StreamFeeder feeder = new StreamFeeder(System.in, mvnProcess.getOutputStream());
			feeder.start();
			StreamPumper errorPumper = new StreamPumper(mvnProcess.getErrorStream(), new SystemOutHandler());
			errorPumper.start();

			return mvnProcess;
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	protected MojoDescriptor getMojoDescriptor(Plugin plugin, String goal) throws MojoExecutionException {
		try {
			return this.pluginManager.getMojoDescriptor(plugin, goal, super.project.getRemotePluginRepositories(),
					super.session.getRepositorySession());
		} catch (PluginNotFoundException | PluginResolutionException | PluginDescriptorParsingException
				| MojoNotFoundException | InvalidPluginDescriptorException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private void executeMojo(MojoDescriptor mojoDescriptor, PluginExecution pluginExec) throws MojoExecutionException {
		// creates a DOM with mojo configuration
		Xpp3Dom xpp3Dom = toXpp3Dom(mojoDescriptor.getMojoConfiguration());

		// merge plugin configuration with mojo configuration
		Xpp3Dom mergedConfig = Xpp3DomUtils.mergeXpp3Dom((Xpp3Dom) pluginExec.getConfiguration(), xpp3Dom);

		// remove not supported parameters from configuration, otherwise build will fail
		filterConfigParameter(mojoDescriptor, mergedConfig);

		MojoExecution exec = new MojoExecution(mojoDescriptor, mergedConfig);

		try {
			this.pluginManager.executeMojo(super.session, exec);
		} catch (PluginConfigurationException | PluginManagerException | MojoFailureException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	/*
	 * static helpers
	 */

	private void filterConfigParameter(MojoDescriptor mojoDescriptor, Xpp3Dom config) {
		List<Integer> paramsToRemove = new ArrayList<>();

		// remove not supported parameters in reverse order to keep index valid
		for (int i = config.getChildCount() - 1; i >= 0; i--) {
			Xpp3Dom parameter = config.getChild(i);

			// check if current parameter is supported by current mojo, if not mark it for removal
			if (!mojoDescriptor.getParameterMap().containsKey(parameter.getName())) {
				paramsToRemove.add(i);

				if (getLog().isDebugEnabled()) {
					logDebug("Removing parameter %s from goal %s configuration.",
							MessageUtils.buffer().strong(parameter.getName()),
							MessageUtils.buffer().strong(mojoDescriptor.getFullGoalName()));
				}
			}
		}

		paramsToRemove.forEach(config::removeChild);
	}

	/**
	 * Converts PlexusConfiguration to a Xpp3Dom.
	 *
	 * @param config the PlexusConfiguration. Must not be {@code null}.
	 * @return the Xpp3Dom representation of the PlexusConfiguration
	 */
	private static Xpp3Dom toXpp3Dom(PlexusConfiguration config) {
		Xpp3Dom result = new Xpp3Dom(config.getName());
		result.setValue(config.getValue(null));
		for (String name : config.getAttributeNames()) {
			result.setAttribute(name, config.getAttribute(name));
		}
		for (PlexusConfiguration child : config.getChildren()) {
			result.addChild(toXpp3Dom(child));
		}
		return result;
	}

	private static String unquote(String executable) {
		if ((executable.startsWith("'") && executable.endsWith("'"))
				|| (executable.startsWith("\"") && executable.endsWith("\""))) {
			return executable.substring(1, executable.length() - 1);
		}
		return executable;
	}
}
