/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.StringUtils;

import com.sap.cds.maven.plugin.util.Platform;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * An abstract base class for all NPM related Mojos.
 */
abstract class AbstractNpmMojo extends AbstractNodejsMojo {

	/**
	 * URL of NPM registry to use.
	 */
	@Parameter(property = "npmRegistry")
	private String npmRegistry;

	// internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = PARAM_NPM_EXECUTABLE, readonly = true)
	private File npmExec;

	protected void executeNpm(File workDir, File outputFile, String... args) throws IOException {
		try (OutputStream fileOutput = new FileOutputStream(outputFile)) {
			Utils.prepareDestination(outputFile, false);
			execute(workDir, getNpmExec(), fileOutput, getEnvironment(), null, args);
		}
	}

	protected String executeNpm(File workDir, int[] exitValues, String... args) throws IOException {
		try (ByteArrayOutputStream fileOutput = new ByteArrayOutputStream()) {
			execute(workDir, getNpmExec(), fileOutput, getEnvironment(), exitValues, args);
			return new String(fileOutput.toByteArray(), StandardCharsets.UTF_8);
		}
	}

	protected String executeNpm(File workDir, String... args) throws IOException {
		return executeNpm(workDir, (int[]) null, args);
	}

	private Map<String, String> getEnvironment() {
		Map<String, String> env = new HashMap<>();

		// use configured NPM registry if available
		if (StringUtils.isNotBlank(this.npmRegistry)) {
			env.put("NPM_CONFIG_REGISTRY", this.npmRegistry);
			logInfo("Using npm repository: %s", this.npmRegistry);
		}

		return env;
	}

	private File getNpmExec() throws IOException {
		if (this.npmExec != null && this.npmExec.canExecute()) {
			logInfo("Using npm from previous installation: %s", this.npmExec);
			return this.npmExec;
		}
		return Utils.findExecutable(Platform.CURRENT.npm, this);
	}
}
