/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import static java.lang.String.format;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.List;
import java.util.Optional;

import org.apache.commons.io.FilenameUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.shared.utils.logging.MessageUtils;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.generator.Cds4jCodegen;
import com.sap.cds.generator.Cds4jCodegen.Result;
import com.sap.cds.generator.Cds4jCodegen.Result.Status;
import com.sap.cds.generator.Configuration;
import com.sap.cds.generator.ConfigurationImpl;
import com.sap.cds.generator.MethodStyle;
import com.sap.cds.generator.util.FileSystem;
import com.sap.cds.generator.util.GeneratedFile;
import com.sap.cds.generator.util.GeneratorMode;
import com.sap.cds.generator.util.ParserMode;
import com.sap.cds.maven.plugin.AbstractCdsMojo;
import com.sap.cds.maven.plugin.util.Utils;
import com.sap.cds.reflect.impl.reader.issuecollector.IssueType;

/**
 * Generate Java Pojos for type-safe access to the CDS model.<br>
 * <br>
 * This goal scans the resource directory for a <STRONG>csn.json</STRONG> file and uses it for Java source code
 * generation. It also adds the code output directory to the project's source code directories configuration. This
 * allows you to use the generated Java classes in your custom code without any additional configuration in the
 * project.<br>
 * <br>
 *
 * @since 1.7.0
 */
@Mojo(name = "generate", defaultPhase = LifecyclePhase.GENERATE_RESOURCES, aggregator = true)
public class GenerateMojo extends AbstractCdsMojo {

	/**
	 * Base package of generated Java classes.
	 */
	@Parameter(property = "cds.generate.basePackage")
	private String basePackage;

	/**
	 * The output directory for generated Java classes. This directory is added to the source directories of your
	 * project.
	 */
	@Parameter(property = "cds.codeOutputDirectory", defaultValue = "${project.basedir}/src/gen/", required = true)
	private File codeOutputDirectory;

	/**
	 * Location of the csn file used for Java source code generation. If not specified, all resource directories are
	 * scanned for a <code>csn.json</code> file. If multiple <code>csn.json</code> files are found, the first one is
	 * used.
	 *
	 * @since 1.8.0
	 */
	@Parameter(property = "cds.generate.csnFile")
	private File csnFile;

	/**
	 * Determine whether to generate interfaces extending {@link com.sap.cds.services.EventContext} for actions and
	 * functions.
	 */
	@Parameter(property = "cds.generate.eventContext", defaultValue = "true")
	private Boolean eventContext;

	/**
	 * Namespaces/definitions of CDS model to be excluded from source code generation. From the set of all included
	 * definitions, the following namespaces/definitions will be excluded. An exclude always overrules an include.<br>
	 * For example, the exclude <STRONG>my.bookshop.*</STRONG> will exclude all definitions with namespace my.bookshop
	 * and the exclude <STRONG>my.bookshop.**</STRONG> will exclude all definitions with fully qualified name beginning
	 * with my.bookshop.
	 */
	@Parameter(property = "cds.generate.excludes", defaultValue = "localized.**")
	private List<String> excludes;

	/**
	 * Namespaces/definitions of CDS model to be included into source code generation. By default, all definitions/namespaces
	 * are included unless an includes is specified explicitly. In which case, only the explicitly specified
	 * definitions/namespaces are included.<br>
	 * For example, the include <STRONG>my.bookshop.orders.*</STRONG> will include only those definitions with namespace my.bookshop.orders and
	 * the include <STRONG>my.bookshop.books.**</STRONG> will include all definitions with fully qualified name beginning with my.bookshop.books.
	 *
	 * @since 1.21.0
	 */
	@Parameter(property = "cds.generate.includes")
	private List<String> includes;

	/**
	 * Defines the generator mode.<br>
	 * Valid values: {@link GeneratorMode#TOLERANT TOLERANT} or {@link GeneratorMode#STRICT STRICT}.
	 */
	@Parameter(property = "cds.generate.generatorMode")
	private GeneratorMode generatorMode;

	/**
	 * Defines method styling for accessor interface methods; follows either the Java Bean style method naming for
	 * getters & setters denoted by {@link MethodStyle#BEAN BEAN} or the Fluent style denoted by
	 * {@link MethodStyle#FLUENT FLUENT}.<br>
	 * Valid values: {@link MethodStyle#BEAN BEAN} or {@link MethodStyle#FLUENT FLUENT}.
	 */
	@Parameter(property = "cds.generate.methodStyle")
	private MethodStyle methodStyle;

	/**
	 * Defines the parser mode.<br>
	 * Valid values: {@link ParserMode#TOLERANT TOLERANT} or {@link ParserMode#STRICT STRICT}.
	 */
	@Parameter(property = "cds.generate.parserMode")
	private ParserMode parserMode;

	/**
	 * Skip execution of this goal.
	 */
	@Parameter(property = "cds.generate.skip", defaultValue = "false")
	private boolean skip;

	/**
	 * Determine whether to generate Javadoc for the generated interfaces.
	 *
	 * @since 1.17.0
	 */
	@Parameter(property = "cds.documentation", defaultValue = "true")
	private boolean documentation;

	// just for test purpose
	private boolean addedSourceDir = false;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			// always add Pojo generation directory to source directories, otherwise Eclipse and VScode will not find
			// Pojos in classpath
			addSourceDir();

			// during a Maven build it's always true, in Eclipse m2e context it can be false
			if (super.buildContext.hasDelta(super.project.getBasedir())) {
				generate();
			} else {
				logInfo("No changes in artifact %s, skipping execution.", super.project.getArtifactId());
			}
		} else {
			logInfo("Skipping execution.");
		}
	}

	/**
	 * Just for testing purpose.
	 *
	 * @return <code>true</code> if source dir was added during execution.
	 */
	@VisibleForTesting
	boolean isAddedSourceDir() {
		return this.addedSourceDir;
	}

	private void addSourceDir() {
		// Cds4j generator adds a Java subdirectory
		File javaSourceDir = new File(this.codeOutputDirectory, "java");

		Optional<String> srcDirOpt = super.project.getCompileSourceRoots().stream() //
				.filter(srcDir -> { // filter out different paths
					logDebug("Found source directory %s in project configuration.", MessageUtils.buffer().strong(srcDir));
					return FilenameUtils.equalsNormalizedOnSystem(srcDir, javaSourceDir.getAbsolutePath());
				}).findFirst();

		// adding same source dir multiple times causes compiler to complain about already existing classes
		if (!srcDirOpt.isPresent()) {
			logInfo("Adding code output directory %s to source directories.", MessageUtils.buffer().strong(javaSourceDir));
			super.project.addCompileSourceRoot(javaSourceDir.getAbsolutePath());
			this.addedSourceDir = true;
		} else {
			logInfo("Code output directory %s is already in source directories.", MessageUtils.buffer().strong(javaSourceDir));
		}
	}

	private void generate() throws MojoExecutionException {
		// check if csn file is configured or needs to be located
		if (this.csnFile == null) {
			this.csnFile = lookupCsnFile();
		}

		if (this.csnFile != null && this.csnFile.exists()) {
			logInfo("Using %s to generate Java classes into %s.", MessageUtils.buffer().strong(this.csnFile),
					MessageUtils.buffer().strong(this.codeOutputDirectory));

			// generate Java POJOs: use {@link Cds4jCodegen} to generate Java classes for model access
			generateClasses(this.csnFile, this.codeOutputDirectory);

			// notify m2e about changes on filesystem
			super.buildContext.refresh(this.codeOutputDirectory);
		} else {
			logInfo("No csn file found, skipping execution.");
		}
	}

	/**
	 * Generate Java Pojos: use {@link Cds4jCodegen} to generate Java classes for model access.
	 *
	 * @throws MojoExecutionException if code generation failed.
	 */
	private void generateClasses(File csnJsonFile, File genOutDir) throws MojoExecutionException {
		GeneratedFile.Accessor fs = new FileSystem(genOutDir.toPath(), true);

		Configuration codeGenCfg = getConfiguration();

		try {
			Result result = new Cds4jCodegen(codeGenCfg).generate(() -> Files.readAllBytes(csnJsonFile.toPath()), fs);

			reportCodegenIssues(result, codeGenCfg);

			if (result.getStatus() == Status.SUCCESS) {
				logInfo("Class generation finished successfully");
			} else {
				throw new MojoExecutionException("Failed to generate Java classes");
			}
		} catch (IOException e) {
			throw new MojoExecutionException("Failed to generate Java classes: ", e);
		}
	}

	/**
	 * @return a {@link Configuration} with current settings.
	 */
	private Configuration getConfiguration() {
		ConfigurationImpl cfg = new ConfigurationImpl();
		Utils.setIfNotNull(cfg::setParserMode, this.parserMode);
		Utils.setIfNotNull(cfg::setGeneratorMode, this.generatorMode);
		Utils.setIfNotNull(cfg::setMethodStyle, this.methodStyle);
		Utils.setIfNotNull(cfg::setBasePackage, this.basePackage);
		Utils.setIfNotNull(cfg::setEventContext, this.eventContext != null ? this.eventContext.toString() : null);
		Utils.setIfNotNull(cfg::setDocs, this.documentation);
		Utils.setIfNotNull(cfg::setExcludes, this.excludes);
		Utils.setIfNotNull(cfg::setIncludes, this.includes);
		return cfg;
	}

	private File lookupCsnFile() {
		return Utils.getResourceDirs(super.project).stream() // just scan within the resource directories
				.map(resourceDir -> scanDirectory(resourceDir, "**/csn.json")) //
				.flatMap(List<File>::stream) // put all csn files into one list
				.findFirst().orElse(null); // get first element
	}

	private void reportCodegenIssues(Result result, Configuration codeGenCfg) {
		result.getIssues().stream().forEach(issue -> {
			String msg = format("[%s] %s : %s", issue.getType(), issue.getLocation(), issue.getMessage());
			if ((issue.getType() == IssueType.CRITICAL)
					|| (issue.getType() == IssueType.ERROR && codeGenCfg.getGeneratorMode() == GeneratorMode.STRICT)) {
				getLog().error(msg);
			} else {
				getLog().warn(msg);
			}
		});
	}
}
