/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.shared.utils.StringUtils;
import org.apache.maven.shared.utils.logging.MessageUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Install <strong><a href="https://www.npmjs.com/package/@sap/cds-dk">@sap/cds-dk</a></strong> in the current CAP Java
 * project.<br>
 * By default, this goal looks for an already installed <strong>@sap/cds-dk</strong> and skips installation if it was
 * found. It doesn't validate the found version against the requested version to install and the installed
 * <strong>@sap/cds-dk</strong> could be outdated. Add property <code>-Dcds.install-cdsdk.force=true</code> to the Maven
 * command line to force the installation of a <strong>@sap/cds-dk</strong> in the configured version.<br>
 * <br>
 *
 * @since 1.7.0
 */
@Mojo(name = "install-cdsdk", defaultPhase = LifecyclePhase.INITIALIZE, aggregator = true)
public class InstallCdsdkMojo extends AbstractNpmMojo {

	private static final String DEFAULT_CDSDK_VERSION = "^4";

	private static final String SAP_CDS_DK = "@sap/cds-dk";

	/**
	 * Defines additional arguments that are passed to the command line that is executed to install the
	 * <strong>@sap/cds-dk</strong>. The arguments are separated by spaces. For example it can be used to enable the
	 * verbose output of {@code npm} with the additional argument {@code --dd}.
	 *
	 * @since 1.23.0
	 */
	@Parameter(property = "cds.install-cdsdk.arguments")
	private String arguments;

	/**
	 * Enables global installation of <strong>@sap/cds-dk</strong>.
	 */
	@Parameter(defaultValue = "false", property = "cds.install-cdsdk.global")
	private boolean global;

	/**
	 * Force installation of <strong>@sap/cds-dk</strong>, even if it's already installed in any version.
	 */
	@Parameter(defaultValue = "false", property = "cds.install-cdsdk.force")
	private boolean force;

	/**
	 * Skip execution of this goal.
	 */
	@Parameter(property = "cds.install-cdsdk.skip", defaultValue = "false")
	private boolean skip;

	/**
	 * Version of <strong><a href="https://www.npmjs.com/package/@sap/cds-dk">@sap/cds-dk</a></strong> to install. The
	 * version value has to be a valid <a href="https://docs.npmjs.com/about-semantic-versioning">semantic version</a>
	 * that is used for NPM modules.
	 */
	@Parameter(defaultValue = DEFAULT_CDSDK_VERSION, required = true, property = "cds.install-cdsdk.version")
	private String version;

	/**
	 * The working directory to be used for installation of <strong>@sap/cds-dk</strong>. If it's not specified, this
	 * goal is using the directory that contains a <strong>.cdsrc.json</strong> or <strong>package.json</strong> file.
	 * The goal goes up the project hierarchy on the filesystem until one of these files is found or the topmost project
	 * directory has been reached.
	 */
	@Parameter
	private File workingDirectory;

	// fields used just for testing purpose
	private boolean executed = false;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			File workDir = getWorkingDirectory();

			if (this.force) {
				install(workDir);
			} else {
				installIfMissing(workDir);
			}
		} else {
			logInfo("Skipping execution.");
		}
	}

	/**
	 * Just for testing purpose.
	 *
	 * @return <code>true</code> if @sap/cds-dk was installed by this goal
	 */
	boolean isExecuted() {
		return this.executed;
	}

	private void install(File workDir) throws MojoExecutionException {
		String cdsdk = SAP_CDS_DK + "@" + this.version;

		try {
			String[] args = null;
			if (this.global) {
				logInfo("Installing %s global", MessageUtils.buffer().strong(cdsdk));
				args = getArguments("install", "-g", cdsdk);
			} else {
				logInfo("Installing %s into: %s", MessageUtils.buffer().strong(cdsdk),
						MessageUtils.buffer().strong(workDir));
				args = getArguments("install", cdsdk, "--no-save");
			}
			logInfo(executeNpm(workDir, args));
			this.executed = true;
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private String[] getArguments(String... args) {
		List<String> allArgs = new ArrayList<>();
		Collections.addAll(allArgs, args);
		if (StringUtils.isNotBlank(this.arguments)) {
			Collections.addAll(allArgs, Utils.splitByWhitespaces(this.arguments));
		}
		return allArgs.toArray(new String[0]);
	}

	private void installIfMissing(File workDir) throws MojoExecutionException {
		boolean isInstalled = this.global ? isInstalledGlobal(workDir) : isInstalledLocal(workDir);

		if (!isInstalled) {
			install(workDir);
		}
	}

	private boolean isInstalledLocal(File workDir) {
		File cdsFile = new File(workDir, "node_modules/.bin/cds");
		// also ensure that the @sap/cds-dk module is installed, the cds cli can also exist without this module. In such
		// a case the "cds build" fails.
		File cdsdkPackageJson = new File(workDir, "node_modules/@sap/cds-dk/package.json");

		boolean isInstalled = cdsFile.canExecute() && cdsdkPackageJson.exists();

		if (isInstalled) {
			logInfo("Found locally installed @sap/cds-dk, skipping execution.");
		}

		return isInstalled;
	}

	private boolean isInstalledGlobal(File workDir) throws MojoExecutionException {
		// also ignore error code 1 as npm still returns a json list with installed modules
		final int[] exitValues = { 0, 1 };

		// get a list of all installed modules, but only the root level (--depth=0)
		String[] args = { "list", "-g", "--json", "--depth=0" };

		try {
			// execute npm list ..., ignore error code 1 as npm still returns a json list with installed modules
			String output = executeNpm(workDir, exitValues, args);

			return parseJsonOutput(output);
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private boolean parseJsonOutput(String output) {
		logDebug(output);

		boolean installed = false;

		try {
			JsonNode outputJson = new ObjectMapper().readTree(output);

			// the property "dependencies" contains a map with installed modules and dependencies of package.json
			JsonNode dependencies = outputJson.get("dependencies");
			if (dependencies != null) {
				// try to get details about @sap/cds-dk
				JsonNode cdsdk = dependencies.get(SAP_CDS_DK);

				// if @sap/cds-dk is part of the dependencies, it's either installed or a dependency in package.json
				if (cdsdk != null) {
					// the property "version" contains the @sap/cds-dk version and it's an
					// additional check to ensure a valid installation
					JsonNode cdsdkVersion = cdsdk.get("version");
					if (cdsdkVersion != null) {
						logInfo("Found globally installed @sap/cds-dk@%s, skipping execution.", cdsdkVersion.asText());
						installed = true;
					} else {
						logDebug("Couldn't get version of globally installed @sap/cds-dk.");
					}
				} else {
					logDebug("Couldn't find a globally installed @sap/cds-dk.");
				}
			}
		} catch (JsonProcessingException e) {
			logDebug(e);
		}
		return installed;
	}

	private File getWorkingDirectory() {
		if (this.workingDirectory == null) {
			// determine cds working directory if not specified
			this.workingDirectory = findCdsWorkingDir();
		} else {
			logInfo("Using configured working directory: %s", this.workingDirectory);
		}
		return this.workingDirectory;
	}
}
