/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.io.IOException;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.StringUtils;

import com.sap.cds.maven.plugin.util.Utils;

/**
 * Execute an <code>npm</code> command on the current CAP Java project.<br>
 * Call <code>mvn com.sap.cds:cds-maven-plugin:npm -Darguments="..."</code> to execute <code>npm</code> command with
 * given arguments.<br>
 * <br>
 *
 * @since 1.7.0
 */
@Mojo(name = "npm", defaultPhase = LifecyclePhase.INITIALIZE, aggregator = true, requiresProject = false)
public class NpmMojo extends AbstractNpmMojo {

	/**
	 * Line of arguments separated by space, which are given to <code>npm</code> for execution. For example: "install -g
	 * ...".
	 */
	@Parameter(property = "arguments")
	private String arguments;

	/**
	 * Standard and error output will be redirected to the file. If not specified, the standard Maven logging is used.
	 */
	@Parameter(property = "outputFile")
	private File outputFile;

	/**
	 * Skip execution of this goal.
	 */
	@Parameter(property = "cds.npm.skip", defaultValue = "false")
	private boolean skip;

	/**
	 * The working directory to be used during <strong>npm</strong> command execution. If it's not specified, the plugin
	 * is using the directory that contains a <strong>.cdsrc.json</strong> or <strong>package.json</strong> file. The
	 * plugin goes up the project hierarchy on the filesystem until one of these files is found or the topmost project
	 * directory has been reached.
	 */
	@Parameter
	private File workingDirectory;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			executeNpm();
		} else {
			logInfo("Skipping execution.");
		}
	}

	private void executeNpm() throws MojoExecutionException {
		try {
			if (this.outputFile != null) {
				executeNpm(getWorkingDirectory(), this.outputFile, getArguments());
			} else {
				logInfo(executeNpm(getWorkingDirectory(), getArguments()));
			}
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private String[] getArguments() throws MojoExecutionException {
		if (StringUtils.isBlank(this.arguments)) {
			throw new MojoExecutionException("No arguments provided.");
		}

		return Utils.splitByWhitespaces(this.arguments);
	}

	private File getWorkingDirectory() {
		// if Mojo is cli executed, use the current directory as working directory
		if (isCliExecuted()) {
			return null;
		}
		if (this.workingDirectory == null) {
			this.workingDirectory = findCdsWorkingDir();
		} else {
			logInfo("Using configured working directory: %s", this.workingDirectory);
		}
		return this.workingDirectory;
	}
}
