package com.sap.cds.maven.plugin.build;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.StringUtils;

import com.sap.cds.maven.plugin.util.Platform;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Execute an <code>npx</code> command on the current CAP Java NG project.<br>
 *
 * @since 1.25.0
 */
@Mojo(name = "npx", defaultPhase = LifecyclePhase.INITIALIZE, aggregator = true, requiresProject = false)
public class NpxMojo extends AbstractNpmMojo {

	/**
	 * A space-delimited line of arguments passed to {@code npx} for execution. For example: {@code cowsay goodbye!}.
	 */
	@Parameter(property = "cds.npx.arguments", required = true)
	private String arguments;

	/**
	 * Configures an optional file to which the standard and error output will be written. If not specified, the standard
	 * Maven logging is used.
	 */
	@Parameter(property = "cds.npx.outputFile")
	private File outputFile;

	/**
	 * Indicates whether execution of this goal should be skipped or not.
	 */
	@Parameter(property = "cds.npx.skip", defaultValue = "false")
	private boolean skip;

	/**
	 * The working directory to be used for <strong>npx</strong> command execution. If it's not specified, the plugin is
	 * using the directory that contains a <strong>.cdsrc.json</strong> or <strong>package.json</strong> file. The plugin
	 * goes up the project hierarchy on the filesystem until one of these files is found or the topmost project directory
	 * has been reached.
	 */
	@Parameter(property = "cds.npx.workingDirectory")
	private File workingDirectory;

	/*
	 * Path to npx executable. internally used parameter, not configurable in pom.xml
	 */
	@Parameter(defaultValue = PARAM_NPX_EXECUTABLE, readonly = true)
	private File npxExec;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			executeNpx();
		} else {
			logInfo("Skipping execution.");
		}
	}

	/**
	 * Gets called to inject parameter "npxExec".
	 *
	 * @param npxExec a {@link File} pointing to npx executable
	 */
	public void setNpxExec(File npxExec) {
		if (npxExec != null && npxExec.canExecute()) {
			this.npxExec = npxExec;
			logInfo("Use npx provided by goal install-node: %s", strong(this.npxExec));
		}
	}

	/**
	 * Gets called to inject parameter "workingDirectory".
	 *
	 * @param workingDirectory a {@link File} pointing to working directory
	 */
	public void setWorkingDirectory(File workingDirectory) {
		if (workingDirectory != null) {
			if (workingDirectory.isDirectory()) {
				this.workingDirectory = workingDirectory;
				logInfo("Use configured working directory: %s", strong(this.workingDirectory));
			} else {
				logWarn("Configured working directory %s isn't a valid directory, can't be used.", strong(workingDirectory));
			}
		}
	}

	private void executeNpx() throws MojoExecutionException {
		try {
			if (this.outputFile != null) {
				try (OutputStream fileOutput = new FileOutputStream(this.outputFile)) {
					execute(getWorkingDirectory(), getNpxExec(), fileOutput, null, null, getArguments());
				}
			} else {
				try (ByteArrayOutputStream fileOutput = new ByteArrayOutputStream()) {
					execute(getWorkingDirectory(), getNpxExec(), fileOutput, null, null, getArguments());
					logInfo(new String(fileOutput.toByteArray(), StandardCharsets.UTF_8));
				}
			}
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private String[] getArguments() throws MojoExecutionException {
		if (StringUtils.isBlank(this.arguments)) {
			throw new MojoExecutionException("No arguments provided.");
		}
		return Utils.splitByWhitespaces(this.arguments);
	}

	private File getNpxExec() throws IOException {
		if (this.npxExec == null) {
			this.npxExec = Utils.findExecutable(Platform.CURRENT.npx, this);
		}
		return this.npxExec;
	}

	private File getWorkingDirectory() {
		if (this.workingDirectory == null) {
			this.workingDirectory = findCdsWorkingDir();
		}
		return this.workingDirectory;
	}
}
