package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.io.IOException;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.util.DirectoryWatcher;
import org.apache.commons.io.FileUtils;
import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.descriptor.MojoDescriptor;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.shared.invoker.InvocationRequest;
import org.apache.maven.shared.invoker.InvocationResult;
import org.apache.maven.shared.utils.cli.ShutdownHookUtils;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import static com.sap.cds.maven.plugin.util.TriggerFileUtil.getTriggerFilePath;

/**
 * Starts a CAP Java NG application and watches for changes in the CDS model. If changes are detected, the application
 * is built and automatically restarted.
 *
 * Call <code>mvn cds:watch</code> or <code>mvn com.sap.cds:cds-maven-plugin:watch</code> to start the CAP Java NG
 * application.<br>
 * <br>
 * In order to get even faster development cycles locally, add an optional dependency to the Spring Boot Developer tools
 * in the pom.xml. In this case the watch goal leaves the restart of the application to the developer tools, which is
 * faster.
 *
 * <pre>
 * {@code <dependencies>
 *   ...
 *   <dependency>
 *     <groupId>org.springframework.boot</groupId>
 *     <artifactId>spring-boot-devtools</artifactId>
 *     <optional>true</optional>
 *     <scope>runtime</scope>
 *   </dependency>
 *   ...
 * </dependencies>}
 * </pre>
 *
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 1.17.0
 */
@Mojo(name = "watch", defaultPhase = LifecyclePhase.NONE, aggregator = true)
public class WatchMojo extends BuildMojo {

	private static final String SRC_MAIN_RESOURCES = "/src/main/resources/";
	private static final String SPRING_BOOT_MAVEN_PLUGIN = "org.springframework.boot:spring-boot-maven-plugin";

	/**
	 * A comma separated list with GLOB pattern describing directories and files to be exclude from watching for changes.
	 * Changes in these directories and files will be ignored.<br>
	 * See <a href=
	 * "https://docs.oracle.com/javase/8/docs/api/java/nio/file/FileSystem.html#getPathMatcher-java.lang.String-">here</a>
	 * for further details about the supported GLOB patterns.
	 */
	@Parameter(property = "cds.watch.excludes", defaultValue = "**/node_modules,**/target,**/gen", required = true)
	private List<String> excludes;

	/**
	 * A comma separated list of GLOB patterns describing files to be include in watching for changes. Changes on those
	 * files will trigger a build and restart of the application.<br>
	 * See <a href=
	 * "https://docs.oracle.com/javase/8/docs/api/java/nio/file/FileSystem.html#getPathMatcher-java.lang.String-">here</a>
	 * for further details about the supported GLOB patterns.
	 */
	@Parameter(property = "cds.watch.includes", defaultValue = "**/*.cds,**/*.csv", required = true)
	private List<String> includes;

	/**
	 * Indicates whether the CAP Java NG application will be started or not. If the application start is disabled, only a
	 * {@code cds:build} is executed on changes in the CDS model.
	 */
	protected boolean noStart;

	// internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${project.build.outputDirectory}", readonly = true, required = true)
	private File outputDir;

	private Process mvnProcess;

	private DirectoryWatcher watcher;

	private boolean useDeveloperTools;

	private volatile Instant lastRestart;

	private Plugin resourcesPlugin;

	@Override
	public void execute() throws MojoExecutionException {
		// ensure this goal is executed from command line.
		ensureCliExecuted();

		initialize();

		if (this.noStart) {
			// initial cds build
			performCdsBuild();
		} else {
			// initial start of the application
			startApplication();
		}

		this.lastRestart = Instant.now();

		try {
			this.watcher.start();

			this.watcher.join();
		} catch (IOException e) {
			logError(e);
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			logDebug(e);
			// as thread was interrupted also stop running application
			stopApplication();
		}
	}

	@VisibleForTesting
	synchronized Instant getLastRestart() {
		return this.lastRestart;
	}

	/**
	 * Gets called when a file has changed.
	 */
	private synchronized void onFileChanged(boolean watchedFileChanged) {
		if (this.useDeveloperTools || this.noStart) {
			// just perform a cds build, restart will be done by developer tools
			if(watchedFileChanged) {
				performCdsBuild();
				this.lastRestart = Instant.now();
			}
			touchTriggerFileIfPresent();
		} else {
			if(watchedFileChanged) {
				// first stop a running application instance
				stopApplication();

				try {
					startApplication();
				} catch (MojoExecutionException e) {
					logError(e);
				}
				this.lastRestart = Instant.now();
			}
		}

	}

	private void touchTriggerFileIfPresent() {

		getTriggerFilePath(this.project.getBasedir().getPath() + SRC_MAIN_RESOURCES).ifPresent(triggerFilePath -> {
			try {
				FileUtils.touch(new File(triggerFilePath));
				logDebug("Touched Spring Boot devtools trigger file: %s", triggerFilePath);
			} catch (IOException e) {
				throw new RuntimeException("Failed to touch devtools trigger file: ", e); //NOSONAR
			}
		});
		logDebug("skipped Spring Boot devtools trigger file.");
	}

	private void performCdsBuild() {
		try {
			// first perform a cds:build ...
			super.executeBuild();

			logDebug("Copying resources with org.apache.maven.plugins:maven-resources-plugin.");

			// ... then copy build result to target/classes folder to make it visible to dev. tools -> triggers restart
			executeGoal(this.resourcesPlugin, "resources");
		} catch (MojoExecutionException e) {
			logWarn(e);
		}
	}

	private void startApplication() throws MojoExecutionException {
		InvocationRequest request = createRequest(getStartGoals());

		logInfo("Starting application %s", super.project.getName());

		this.mvnProcess = executeMaven(request);
	}

	/**
	 * Stops the spring-boot application.
	 *
	 * @return the application stop invocation {@link InvocationResult result}
	 */
	private void stopApplication() {
		if (this.mvnProcess != null) {
			logInfo("Stopping application %s", super.project.getName());

			this.mvnProcess.destroy();

			try {
				boolean exited = this.mvnProcess.waitFor(5, TimeUnit.SECONDS);
				if (exited) {
					logInfo("Stopped application %s: rc=%d", super.project.getName(), this.mvnProcess.exitValue());
				} else {
					this.mvnProcess.destroyForcibly();
					logInfo("Destroyed application %s", super.project.getName());
				}
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				logError(e);
			} finally {
				this.mvnProcess = null;
			}
		}
	}

	private void initialize() throws MojoExecutionException {
		// ensure the spring-boot maven plugin is available
		// TODO: if no-start is declared we don't need the spring boot plugin
		Plugin springBootPlugin = ensurePlugin(SPRING_BOOT_MAVEN_PLUGIN);

		// https://docs.spring.io/spring-boot/docs/current/maven-plugin/reference/htmlsingle/#goals-run-parameters-details-fork
		boolean fork = Boolean.parseBoolean(getParameterValue(springBootPlugin, "run", "fork"));

		// search for developer tools dependency in current pom.xml
		this.useDeveloperTools = hasDependency("org.springframework.boot:spring-boot-devtools:jar");

		if (fork && this.useDeveloperTools) {
			logInfo("Found dependency to Spring-Boot Developer Tools, will use it for a faster restart of the CAP Java NG application.");

			// get maven-resources-plugin for later usage after the cds:build is performed
			this.resourcesPlugin = ensurePlugin("org.apache.maven.plugins:maven-resources-plugin");
		} else {
			logInfo("No Spring-Boot Developer Tools found or they are deactivated.");
		}

		ShutdownHookUtils.addShutDownHook(new Thread(this::stopApplication));

		this.watcher = new DirectoryWatcher(getReactorBaseDirectory().toPath(), this::onFileChanged, this, this.includes, this.excludes);
	}

	/**
	 * Ensures that current project uses the plugin with the given key.
	 *
	 * @param pluginKey the plugin key
	 * @return the {@link Plugin} for given key
	 * @throws MojoExecutionException if project doesn't use plugin with given key
	 */
	private Plugin ensurePlugin(String pluginKey) throws MojoExecutionException {
		Plugin plugin = getPlugin(pluginKey);
		if (plugin == null) {
			throw new MojoExecutionException("Project doesn't use plugin " + pluginKey + ". To watch this project, it's required.");
		}
		return plugin;
	}

	private String getParameterValue(Plugin plugin, String goal, String paramName) throws MojoExecutionException {
		// first get default value from MOJO description for given plugin / goal
		MojoDescriptor mojoDescriptor = getMojoDescriptor(plugin, goal);
		org.apache.maven.plugin.descriptor.Parameter parameter = mojoDescriptor.getParameterMap().get(paramName);
		String defaultValue = parameter != null ? parameter.getDefaultValue() : null;

		// then get current configuration of plugin from pom.xml
		Xpp3Dom cfg = ((Xpp3Dom) plugin.getConfiguration());
		Xpp3Dom forkAttr = cfg.getChild(paramName);

		// return default value if requested parameter is not configured
		return forkAttr != null ? forkAttr.getValue() : defaultValue;
	}

	private Plugin getPlugin(String pluginKey) {
		return super.project.getModel().getBuild().getPluginsAsMap().get(pluginKey);
	}

	private boolean hasDependency(String depKey) {
		return super.project.getDependencies().stream().anyMatch(dependency -> dependency.getManagementKey().equals(depKey));
	}

	private static List<String> getStartGoals() {
		return Collections.singletonList(SPRING_BOOT_MAVEN_PLUGIN + ":run");
	}

}
