/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.StringTokenizer;

import org.apache.commons.io.IOUtils;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.maven.settings.Proxy;
import org.codehaus.plexus.util.StringUtils;

import com.sap.cds.maven.plugin.CdsMojoLogger;

/**
 * Used to download the Node archive from a given download URL.
 */
public class Downloader {

	private final CdsMojoLogger logger;

	private final List<Proxy> proxies;

	/**
	 * Constructs a new {@link Downloader} instance.
	 *
	 * @param logger the required {@link CdsMojoLogger} instance.
	 * @throws NullPointerException if any of the required parameters is <code>null</code>.
	 */
	public Downloader(CdsMojoLogger logger) {
		this(null, logger);
	}

	/**
	 * Constructs a new {@link Downloader} instance.
	 *
	 * @param proxies an optional list of proxies
	 * @param logger  the required {@link CdsMojoLogger} instance.
	 * @throws NullPointerException if any of the required parameters is <code>null</code>.
	 */
	public Downloader(List<Proxy> proxies, CdsMojoLogger logger) {
		this.proxies = proxies != null ? new ArrayList<>(proxies) : Collections.emptyList();
		this.logger = requireNonNull(logger, "logger must not be null");
	}

	/**
	 * Performs a download from given URL.
	 *
	 * @param downloadUrl the URL to download from
	 * @param destination the destination {@link File}
	 * @throws IOException          if download failed
	 * @throws NullPointerException if any of the required parameters is <code>null</code>.
	 */
	public void download(String downloadUrl, File destination) throws IOException {
		download(downloadUrl, destination, null, null);
	}

	/**
	 * Performs a download from given URL by using the given credentials
	 *
	 * @param downloadUrl the URL to download from
	 * @param destination the destination {@link File}
	 * @param userName    an optional user name
	 * @param password    an optional user password
	 * @throws IOException              if download failed
	 * @throws NullPointerException     if any of the required parameters is <code>null</code>.
	 * @throws IllegalArgumentException If the given download URL isn’t valid
	 */
	public void download(String downloadUrl, File destination, String userName, String password) throws IOException {
		requireNonNull(destination, "destination must not be null");

		URI downloadUri = URI.create(requireNonNull(downloadUrl, "downloadUrl must not be null"));

		try (CloseableHttpResponse response = execute(downloadUri, userName, password)) {
			int statusCode = response.getStatusLine().getStatusCode();
			if (statusCode != 200) {
				throw new IOException(format("Got error code %d from the server.", statusCode));
			}

			File destinationParent = destination.getParentFile();
			if (!destinationParent.exists() && !destinationParent.mkdirs())
				throw new IOException(format("Creating %s failed.", destinationParent));

			try (InputStream ris = response.getEntity().getContent();
					FileOutputStream fos = new FileOutputStream(destination)) {
				IOUtils.copy(ris, fos);
			}
		}
	}

	Proxy getProxyForUrl(URI requestUri) {
		if (!this.proxies.isEmpty()) {
			for (Proxy proxy : this.proxies) {
				if (!isNonProxyHost(proxy, requestUri.getHost())) {
					return proxy;
				}
			}
			this.logger.logInfo("Could not find matching proxy for host: %s", requestUri.getHost());
		}
		return null;
	}

	private CloseableHttpResponse execute(URI requestUrl, String userName, String password) throws IOException {
		// check if proxy is configured for the URL
		Proxy proxy = getProxyForUrl(requestUrl);
		if (proxy != null) {
			this.logger.logInfo("Downloading via proxy: %s:%d", proxy.getHost(), proxy.getPort());
			return executeViaProxy(proxy, requestUrl);
		}
		this.logger.logInfo("No proxy was configured, downloading directly.");
		return executeDirectly(requestUrl, userName, password);
	}

	private static CloseableHttpClient buildHttpClient(CredentialsProvider credentialsProvider) {
		return HttpClients.custom().disableContentCompression().useSystemProperties()
				.setDefaultCredentialsProvider(credentialsProvider).build();
	}

	@SuppressWarnings("resource")
	private static CloseableHttpResponse executeDirectly(URI requestUrl, String userName, String password)
			throws IOException {

		CredentialsProvider credentialsProvider = null;
		if (StringUtils.isNotEmpty(userName) && StringUtils.isNotEmpty(password)) {
			credentialsProvider = makeCredentialsProvider(requestUrl.getHost(), requestUrl.getPort(), userName,
					password);
		}

		return buildHttpClient(credentialsProvider).execute(new HttpGet(requestUrl));
	}

	@SuppressWarnings("resource")
	private static CloseableHttpResponse executeViaProxy(Proxy proxy, URI requestUri) throws IOException {
		CloseableHttpClient proxyClient;

		if (StringUtils.isNotEmpty(proxy.getUsername())) {
			CredentialsProvider credentialsProvider = makeCredentialsProvider(proxy.getHost(), proxy.getPort(),
					proxy.getUsername(), proxy.getPassword());
			proxyClient = buildHttpClient(credentialsProvider);
		} else {
			proxyClient = buildHttpClient(null);
		}

		HttpHost proxyHost = new HttpHost(proxy.getHost(), proxy.getPort(), proxy.getProtocol());
		RequestConfig requestConfig = RequestConfig.custom().setProxy(proxyHost).build();

		HttpGet request = new HttpGet(requestUri);
		request.setConfig(requestConfig);

		return proxyClient.execute(request);
	}

	private static boolean isNonProxyHost(Proxy proxy, String host) {
		if (StringUtils.isNotBlank(proxy.getNonProxyHosts())) {
			for (StringTokenizer tokenizer = new StringTokenizer(proxy.getNonProxyHosts(), "|"); tokenizer
					.hasMoreTokens();) {
				String pattern = tokenizer.nextToken().replace(".", "\\.").replace("*", ".*");
				if (host.matches(pattern)) {
					return true;
				}
			}
		}

		return false;
	}

	private static CredentialsProvider makeCredentialsProvider(String host, int port, String username,
			String password) {
		CredentialsProvider credentialsProvider = new BasicCredentialsProvider();
		credentialsProvider.setCredentials(new AuthScope(host, port),
				new UsernamePasswordCredentials(username, password));
		return credentialsProvider;
	}
}