/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.DefaultExecutor;
import org.apache.commons.exec.ExecuteException;
import org.apache.commons.exec.Executor;
import org.apache.commons.exec.PumpStreamHandler;
import org.apache.commons.exec.ShutdownHookProcessDestroyer;

import com.sap.cds.maven.plugin.CdsMojoLogger;

/**
 * A class to execute subprocesses with a given command line in a specified working directory.
 */
public class ProcessExecutor {
	/**
	 * A default list with exit values of processes to be considered successful
	 */
	private static final int[] DEFAULT_EXIT_VALUES = new int[] { 0 };

	private CommandLine commandLine;

	private final Map<String, String> environment;

	private final Executor executor;

	private int[] exitValues = DEFAULT_EXIT_VALUES;

	/**
	 * Constructs new {@link ProcessExecutor} instance.
	 *
	 * @param workingDir the working directory of the process to run
	 * @param cmdLine    the command line to execute
	 */
	public ProcessExecutor(File workingDir, CommandLine cmdLine) {
		this(workingDir, cmdLine, null, null);
	}

	/**
	 * Constructs new {@link ProcessExecutor} instance.
	 *
	 * @param workingDir the working directory of the process to run
	 * @param cmdLine    the command line to execute
	 * @param env        additional environment variables
	 * @param paths      a list of directories that will be added to path env. variable
	 */
	public ProcessExecutor(File workingDir, CommandLine cmdLine, Map<String, String> env, List<String> paths) {
		this.commandLine = Objects.requireNonNull(cmdLine, "cmdLine");
		this.executor = createExecutor(workingDir);
		this.environment = createEnvironment(paths, env);
	}

	/**
	 * @param outputStream    the output stream to write console output to
	 * @param errOutputStream the error output stream
	 * @param logger          a required {@link CdsMojoLogger}
	 * @return the return code
	 * @throws IOException      if execution failed
	 * @throws ExecuteException execution of subprocess failed or the subprocess returned an exit value indicating a
	 *                          failure {@link Executor#setExitValue(int)}.
	 */
	public int execute(OutputStream outputStream, OutputStream errOutputStream, CdsMojoLogger logger)
			throws IOException {
		this.executor.setStreamHandler(new PumpStreamHandler(outputStream, errOutputStream));
		// set list with exit value of the process to be considered successful
		this.executor.setExitValues(this.exitValues);

		logger.logDebug("Process environment: %s", this.environment);
		return this.executor.execute(this.commandLine, this.environment);
	}

	/**
	 * Executes the process.
	 *
	 * @param logger a required {@link CdsMojoLogger}
	 * @return the console output of executed process
	 * @throws IOException      if process execution failed
	 * @throws ExecuteException execution of subprocess failed or the subprocess returned an exit value indicating a
	 *                          failure {@link Executor#setExitValue(int)}.
	 */
	public String executeAndGetResult(CdsMojoLogger logger) throws IOException {
		ByteArrayOutputStream stdout = new ByteArrayOutputStream();
		try (OutputStream tmp = stdout) {
			this.executor.setStreamHandler(new PumpStreamHandler(tmp));
			// set list with exit value of the process to be considered successful
			this.executor.setExitValues(this.exitValues);
			this.executor.execute(this.commandLine, this.environment);
			return stdout.toString(StandardCharsets.UTF_8.name()).trim();
		} catch (IOException e) {
			logger.logError(new String(stdout.toByteArray(), StandardCharsets.UTF_8));
			throw e;
		}
	}

	public void setExitValues(int[] exitValues) {
		if (exitValues == null || exitValues.length == 0) {
			this.exitValues = DEFAULT_EXIT_VALUES;
		} else {
			this.exitValues = exitValues; // NOSONAR
		}
	}

	private static Map<String, String> createEnvironment(List<String> paths,
			Map<String, String> additionalEnvironment) {
		Map<String, String> environment = new HashMap<>(System.getenv());
		String pathVarName = "PATH";
		String pathVarValue = environment.get(pathVarName);

		if (Platform.CURRENT.isWindows()) {
			for (Map.Entry<String, String> entry : environment.entrySet()) {
				if ("PATH".equalsIgnoreCase(entry.getKey())) {
					pathVarName = entry.getKey();
					pathVarValue = entry.getValue();
					break;
				}
			}
		}

		StringBuilder pathBuilder = new StringBuilder();
		if (pathVarValue != null) {
			pathBuilder.append(pathVarValue).append(File.pathSeparator);
		}

		if (paths != null) {
			for (String path : paths) {
				pathBuilder.insert(0, File.pathSeparator).insert(0, path);
			}
		}
		environment.put(pathVarName, pathBuilder.toString());

		if (additionalEnvironment != null) {
			environment.putAll(additionalEnvironment);
		}

		return environment;
	}

	private static Executor createExecutor(File workingDirectory) {
		DefaultExecutor executor = new DefaultExecutor();
		executor.setWorkingDirectory(workingDirectory);
		executor.setProcessDestroyer(new ShutdownHookProcessDestroyer());

		return executor;
	}
}