/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This class represents a semantic version information.
 *
 * For further details see <a href="https://semver.org">Semantic Versioning</a>.
 */
public class Semver {

	// taken from: https://semver.org
	private static final Pattern SEMVER_REGEX = Pattern.compile(
			"^(0|[1-9]\\d*)\\.(0|[1-9]\\d*)\\.(0|[1-9]\\d*)(?:-((?:0|[1-9]\\d*|\\d*[a-zA-Z-][0-9a-zA-Z-]*)(?:\\.(?:0|[1-9]\\d*|\\d*[a-zA-Z-][0-9a-zA-Z-]*))*))?(?:\\+([0-9a-zA-Z-]+(?:\\.[0-9a-zA-Z-]+)*))?$");

	private final int major;

	private final int minor;

	private final int patch;

	/**
	 * Constructs a new {@link Semver} instance with given version string.
	 *
	 * @param version a version string in a valid semver pattern, e.g. "1.2.4"
	 * @throws IllegalArgumentException if the given version is not a valid Semantic Version with at least major, minor
	 *                                  and patch information.
	 */
	public Semver(String version) {
		Matcher matcher = SEMVER_REGEX.matcher(Objects.requireNonNull(version, "version is null."));

		if (matcher.matches() && matcher.groupCount() >= 3) {
			try {
				this.major = Integer.parseInt(matcher.group(1));
				this.minor = Integer.parseInt(matcher.group(2));
				this.patch = Integer.parseInt(matcher.group(3));
			} catch (NumberFormatException e) {
				throw new IllegalArgumentException(String.format("Not a valid Semantic Version: \"%s\"", version), e);
			}
		} else {
			throw new IllegalArgumentException("Not a valid Semantic Version: " + version);
		}
	}

	/**
	 * Compares this {@link Semver} instance with another
	 *
	 * @param another the other {@link Semver} instance to compare to
	 * @return 0 if both are equal, 1 if this instance is greater, otherwise -1.
	 */
	public int compareTo(Semver another) {
		int result = compareField(another.major, this.major);
		if (result != 0) {
			return result;
		}

		result = compareField(another.minor, this.minor);
		if (result != 0) {
			return result;
		}

		return compareField(another.patch, this.patch);
	}

	@Override
	public String toString() {
		return this.major + "." + this.minor + "." + this.patch;
	}

	private static int compareField(int another, int thiss) {
		if (another == thiss) {
			return 0;
		}
		return thiss < another ? -1 : 1;
	}
}
