/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import java.io.File;
import java.io.IOException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.util.PomUtils;
import com.sap.cds.maven.plugin.util.Utils;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * Add a target platform dependency (<code>cds-starter-${targetPlatform}</code>) to the service module dependencies of
 * the CAP Java project. If the dependency already exists, nothing is done.<br>
 * Call {@code mvn cds:addTargetPlatform -DtargetPlatform=cloudfoundry} or
 * {@code mvn com.sap.cds:cds-maven-plugin:addTargetPlatform -DtargetPlatform=cloudfoundry} to add CloudFoundry target
 * platform.<br>
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 1.5.0
 */
@Mojo(name = "addTargetPlatform", aggregator = true)
public class AddTargetPlatformMojo extends AbstractAddMojo {

	private boolean added;

	/**
	 * Specify the name of the target platform to add.<br>
	 * Allowed values are:
	 * <ul>
	 * <li><strong>cloudfoundry</strong>: adds support for CloudFoundry</li>
	 * </ul>
	 */
	@Parameter(property = "targetPlatform", required = true)
	private String targetPlatform;

	@Override
	public void execute() throws MojoExecutionException, MojoFailureException {
		ensureCliExecuted();

		MavenProject srvProject = findSrvProject();

		updatePom(srvProject.getFile());
	}

	@VisibleForTesting
	boolean isAdded() {
		return this.added;
	}

	private void updatePom(File pomFile) throws MojoExecutionException {
		this.targetPlatform = this.targetPlatform.trim();

		String artifactId = null;
		if (this.targetPlatform.equalsIgnoreCase("cloudfoundry") || this.targetPlatform.equalsIgnoreCase("cf")) {
			artifactId = "cds-starter-cloudfoundry";
		}

		if (artifactId == null) {
			throw new MojoExecutionException("Unknown target platform: " + this.targetPlatform);
		}

		try {
			Document doc = Utils.parse(pomFile);

			this.added = PomUtils.addDependency(doc, CDS_SERVICES_GROUPID, artifactId, null);
			if (this.added) {
				logInfo("Added dependency: %s", artifactId);
				Utils.store(pomFile, doc);
			} else {
				logWarn("Dependency %s already present, nothing added.", artifactId);
			}
		} catch (ParserConfigurationException | SAXException | IOException | TransformerException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}
}
