/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.FileUtils;

import com.sap.cds.maven.plugin.AbstractCdsMojo;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Delete generated files and directories of the previous build from the CAP Java project.<br>
 * Call {@code mvn cds:clean} or {@code mvn com.sap.cds:cds-maven-plugin:clean} to execute cleaning from the command
 * line.<br>
 * <br>
 * This goal deletes the following files and directories from the resource directory of the CAP Java project:
 * <ul>
 * <li>file: <strong>src/main/resources/&#42;&#42;/csn.json</strong></li>
 * <li>file: <strong>src/main/resources/&#42;&#42;/schema.sql</strong></li>
 * <li>directory: <strong>src/main/resources/&#42;&#42;/edmx/</strong></li>
 * <li>directory: <strong>${codeOutputDirectory}</strong></li>
 * </ul>
 * <br>
 *
 * @since 1.7.0
 */
@Mojo(name = "clean", defaultPhase = LifecyclePhase.CLEAN, aggregator = true)
public class CleanMojo extends AbstractCdsMojo {

	private static final String DELETING = "Deleting %s";

	/**
	 * Define the output directory for generated Java source code (POJOs).
	 */
	@Parameter(property = "cds.codeOutputDirectory", defaultValue = "${project.basedir}/src/gen/", required = true)
	private File codeOutputDirectory;

	/**
	 * Skip cleaning of generated files and directories.
	 */
	@Parameter(property = "cds.clean.skip", defaultValue = "${maven.clean.skip}")
	private boolean skip;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			clean();
		} else {
			logInfo("Skipping execution.");
		}
	}

	/**
	 * Clean generated files and classes of previous build.
	 *
	 * @throws MojoExecutionException if cleaning failed
	 */
	private void clean() throws MojoExecutionException {
		try {
			if (this.codeOutputDirectory.exists()) {
				logInfo(DELETING, this.codeOutputDirectory);
				// delete generated Pojos of previous build
				FileUtils.forceDelete(this.codeOutputDirectory);
			}

			List<File> resourceDirs = Utils.getResourceDirs(this.project).collect(Collectors.toList());

			// delete generated csn.json of previous build from resources directories
			File csnJson = Utils.findFile("csn.json", resourceDirs);
			if (csnJson != null) {
				logInfo(DELETING, csnJson);
				FileUtils.forceDelete(csnJson);
			}

			// delete generated edmx files of previous build from resources directories
			File edmxDir = Utils.findDirectory("edmx", resourceDirs);
			if (edmxDir != null) {
				logInfo(DELETING, edmxDir);
				FileUtils.forceDelete(edmxDir);
			}

			// delete generated schema.sql of previous build from resources directories
			File schemaSql = Utils.findFile("schema.sql", resourceDirs);
			if (schemaSql != null) {
				logInfo(DELETING, schemaSql);
				FileUtils.forceDelete(schemaSql);
			}
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}
}
