/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import static java.lang.String.format;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import org.apache.commons.io.FileUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.settings.Proxy;
import org.apache.maven.settings.Server;
import org.codehaus.plexus.util.StringUtils;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.CdsMojoLogger;
import com.sap.cds.maven.plugin.util.ArchiveUtils;
import com.sap.cds.maven.plugin.util.Downloader;
import com.sap.cds.maven.plugin.util.Platform;
import com.sap.cds.maven.plugin.util.Utils;

class NodeInstaller {
	private static final String EXEC_NOT_FOUND = "Executable '%s' not found in Node.js installation.";

	static final String DEFAULT_NODEJS_DOWNLOAD_ROOT = "https://nodejs.org/dist/";

	private String downloadRoot;

	private boolean force = false;

	private File installDirectory;

	private final CdsMojoLogger logger;

	private final NodeCacheResolver nodeResolver;

	private final String nodeVersion;

	private List<Proxy> proxies = Collections.emptyList();

	private File nodeExecFile;

	private File npmExecFile;

	private File npxExecFile;

	private Server server;

	NodeInstaller(String nodeVersion, NodeCacheResolver nodeResolver, CdsMojoLogger logger) {
		this.nodeResolver = Objects.requireNonNull(nodeResolver, "nodeResolver");
		this.nodeVersion = Objects.requireNonNull(nodeVersion, "nodeVersion");
		this.logger = Objects.requireNonNull(logger, "logger");
	}

	File install() throws MojoExecutionException {
		prepareInstall();

		if (this.nodeExecFile.canExecute() && this.npmExecFile.canExecute() && this.npxExecFile.canExecute()
				&& !this.force) {
			this.logger.logInfo("Node.js %s already installed.", this.nodeVersion);
		} else {
			this.logger.logInfo("Installing Node.js %s", this.nodeVersion);
			installNode();
		}
		return this.installDirectory;
	}

	void setDownloadRoot(String downloadRoot) {
		if (downloadRoot != null && !downloadRoot.endsWith("/")) {
			this.downloadRoot = downloadRoot + "/";
		} else {
			this.downloadRoot = downloadRoot;
		}
	}

	void setForce(boolean force) {
		this.force = force;
	}

	void setInstallDirectory(File installDirectory) {
		this.installDirectory = installDirectory;
	}

	void setProxies(List<Proxy> proxies) {
		this.proxies = proxies != null ? proxies : Collections.emptyList();
	}

	void setServer(Server server) {
		this.server = server;
	}

	/**
	 * @return the node executable as {@link File}.
	 */
	File getNodeExec() {
		return this.nodeExecFile;
	}

	/**
	 * @return the npm executable as {@link File}.
	 */
	File getNpmExec() {
		return this.npmExecFile;
	}

	/**
	 * @return the npx executable as {@link File}.
	 */
	File getNpxExec() {
		return this.npxExecFile;
	}

	private void downloadIfMissing(File archive) throws IOException {
		if (!archive.exists() || this.force) {
			FileUtils.deleteQuietly(archive);
			String downloadUrl = this.downloadRoot + Platform.CURRENT.getDownloadPath(this.nodeVersion);
			this.logger.logInfo("Downloading %s to %s", downloadUrl, archive);
			new Downloader(this.proxies, this.server, this.logger).download(downloadUrl, archive);
		} else {
			this.logger.logInfo("Download not required, using Node.js from local cache: %s", archive);
		}
	}

	private void extractFile(File archive) throws IOException {
		// extract archive if installation directory can be locked
		Utils.runIfLockable(new File(this.installDirectory, ".lock"), () -> {
			this.logger.logInfo("Unpacking %s into %s", archive, this.installDirectory);
			try {
				FileUtils.cleanDirectory(this.installDirectory);
				ArchiveUtils.extract(archive, this.installDirectory);

				// set execution flags on non Windows platforms
				if (!Platform.CURRENT.isWindows()) {
					Utils.setExecutionFlag(this.nodeExecFile);
					Utils.setExecutionFlag(this.npmExecFile);
					Utils.setExecutionFlag(this.npxExecFile);
				}
			} catch (IOException e) {
				this.logger.logError(e);
			}
		}, 30);
	}

	private void installNode() throws MojoExecutionException {
		File archive = this.nodeResolver.resolve(this.nodeVersion);

		try {
			downloadIfMissing(archive);

			extractFile(archive);

			validateInstallation();
		} catch (IOException e) {
			// the download or extracting was probably interrupted and archive file is incomplete:
			// delete it to retry from scratch
			this.logger.logError("The archive %s is corrupted and will be deleted. Please try the build again.",
					archive.getPath());
			FileUtils.deleteQuietly(archive);
			FileUtils.deleteQuietly(this.installDirectory);
			throw new MojoExecutionException("Couldn't install Node", e);
		}
	}

	@VisibleForTesting
	void validateInstallation() throws FileNotFoundException {
		if (!this.nodeExecFile.canExecute()) {
			throw new FileNotFoundException(format(EXEC_NOT_FOUND, this.nodeExecFile));
		}
		if (!this.npmExecFile.canExecute()) {
			throw new FileNotFoundException(format(EXEC_NOT_FOUND, this.npmExecFile));
		}
		if (!this.npxExecFile.canExecute()) {
			throw new FileNotFoundException(format(EXEC_NOT_FOUND, this.npxExecFile));
		}
	}

	private void prepareInstall() throws MojoExecutionException {
		if (StringUtils.isBlank(this.downloadRoot)) {
			this.downloadRoot = DEFAULT_NODEJS_DOWNLOAD_ROOT;
		}

		// if an installation directory wasn’t set, use Node cache as installation directory
		if (this.installDirectory == null) {
			this.installDirectory = this.nodeResolver.resolveUnpacked(this.nodeVersion);
		}
		if (!this.installDirectory.exists() && !this.installDirectory.mkdirs()) {
			throw new MojoExecutionException("Failed to create directories " + this.installDirectory);
		}

		this.nodeExecFile = new File(this.installDirectory, Platform.CURRENT.getNodePath(this.nodeVersion));
		this.npmExecFile = new File(this.installDirectory, Platform.CURRENT.getNpmPath(this.nodeVersion));
		this.npxExecFile = new File(this.installDirectory, Platform.CURRENT.getNpxPath(this.nodeVersion));
	}
}
