/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.util.Map;
import java.util.Objects;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.shared.dependency.graph.DependencyGraphBuilder;
import org.apache.maven.shared.utils.logging.MessageBuilder;
import org.apache.maven.shared.utils.logging.MessageUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.sap.cds.maven.plugin.util.DependencyFinder;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Prints detailed version information about the CAP Java project on the console.<br>
 * Call {@code cds:version} or {@code mvn com.sap.cds:cds-maven-plugin:version} to get detailed version information.<br>
 * <br>
 * This goal shows the following version information about the CAP Java project and build environment:
 * <ul>
 * <li>version of CAP Java SDK including CDS4J</li>
 * <li>console output of command line <code>cds version</code></li>
 * <li>version and location of Apache Maven</li>
 * <li>version and location of Java runtime</li>
 * <li>OS information</li>
 * </ul>
 * <br>
 *
 * @since 1.19.0
 */
@Mojo(name = "version", defaultPhase = LifecyclePhase.GENERATE_SOURCES, aggregator = true)
public class VersionMojo extends AbstractCdsCliMojo {

	private static final String DEFAULT_BOUNDARY = "================================================================================";
	private static final String ARCH = "arch";
	private static final String HOME = "home";
	private static final String OS = "OS";
	private static final String JAVA = "Java";
	private static final String JAVA_SDK = "CAP Java SDK";
	private static final String CDS4J = "cds4j";
	private static final String SPRING_BOOT = "Spring Boot";
	private static final String MAVEN = "Maven";
	private static final String UNKNOWN = "unknown";
	private static final String VERSION = "version";

	/**
	 * Skip execution of this goal.
	 */
	@Parameter(property = "cds.version.skip", defaultValue = "false")
	private boolean skip;

	/**
	 * Exclude cds versions from version output.
	 */
	@Parameter(property = "cds.version.excludeCds", defaultValue = "false")
	private boolean excludeCds;

	/**
	 * Indicates whether the version output is in JSON or plain text.
	 */
	@Parameter(property = "cds.version.json", defaultValue = "false")
	private boolean json;

	/**
	 * The begin and end boundary around the version output. After the begin boundary, a new line is printed before the
	 * version output starts. The boundaries are helpful to indicate the location of the version output within the
	 * whole Maven console output, if it's parsed by another application.
	 */
	@Parameter(property = "cds.version.boundary", defaultValue = DEFAULT_BOUNDARY)
	private String boundary;

	@Component(role = DependencyGraphBuilder.class)
	private DependencyGraphBuilder depGraphBuilder;

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${maven.home}", readonly = true)
	private String mavenHome;

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${maven.version}", readonly = true)
	private String mavenVersion;

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${java.home}", readonly = true)
	private String javaHome;

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${java.version}", readonly = true)
	private String javaVersion;

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${os.arch}", readonly = true)
	private String osArch;

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${os.name}", readonly = true)
	private String osName;

	// Internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${os.version}", readonly = true)
	private String osVersion;

	private DependencyFinder finder;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			this.finder = scanDependencyGraph();

			String cdsVersion = !this.excludeCds ? executeCdsVersion() : null;

			if (this.json) {
				printJson(cdsVersion);
			} else {
				printPlain(cdsVersion);
			}
		} else {
			logInfo("Skipping execution.");
		}
	}

	@Override
	protected String executeCdsVersion() {
		try {
			return super.executeCdsVersion();
		} catch (MojoExecutionException e) {
			logDebug(e);
			return null;
		}
	}

	private void printJson(String cdsVersion) throws MojoExecutionException {
		ObjectMapper mapper = new ObjectMapper();

		ObjectNode capJavaSdkJson = mapper.createObjectNode();
		capJavaSdkJson.put(VERSION, getCdsServicesVersion());

		ObjectNode cds4jJson = mapper.createObjectNode();
		cds4jJson.put(VERSION, getCds4jVersion());

		ObjectNode springBootJson = mapper.createObjectNode();
		springBootJson.put(VERSION, getSpringBootVersion());

		ObjectNode mavenJson = mapper.createObjectNode();
		mavenJson.put(VERSION, this.mavenVersion);
		mavenJson.put(HOME, this.mavenHome);

		ObjectNode javaJson = mapper.createObjectNode();
		javaJson.put(VERSION, this.javaVersion);
		javaJson.put(HOME, this.javaHome);

		ObjectNode osJson = mapper.createObjectNode();
		osJson.put("name", this.osName);
		osJson.put(VERSION, this.osVersion);
		osJson.put(ARCH, this.osArch);

		ObjectNode versionJson = mapper.createObjectNode();
		versionJson.set(JAVA_SDK, capJavaSdkJson);
		versionJson.set(CDS4J, cds4jJson);
		versionJson.set(SPRING_BOOT, springBootJson);
		versionJson.set(MAVEN, mavenJson);
		versionJson.set(JAVA, javaJson);
		versionJson.set(OS, osJson);

		if (cdsVersion != null) {
			// add all cds components to JSON output
			Map<String, String> cdsVersions = Utils.parseCdsdkVersionOutput(cdsVersion);
			cdsVersions.forEach((key, value) -> {
				if (key.startsWith("@sap/")) {
					ObjectNode cdsJson = mapper.createObjectNode();
					cdsJson.put(VERSION, value);
					versionJson.set(key, cdsJson);
				}
			});

			// add home location of @sap/cds to corresponding JSON object
			ObjectNode cdsJson = (ObjectNode) versionJson.get("@sap/cds");
			if (cdsJson != null && cdsVersions.containsKey(HOME)) {
				cdsJson.put(HOME, cdsVersions.get(HOME));
			}
		}

		try {
			String jsonOutput = mapper.writerWithDefaultPrettyPrinter().writeValueAsString(versionJson);
			StringBuilder builder = new StringBuilder(2048);
			builder.append("\n").append(this.boundary).append("\n").append(jsonOutput).append("\n")
					.append(this.boundary);
			logInfo(builder.toString());
		} catch (JsonProcessingException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private void printPlain(String cdsVersion) {
		MessageBuilder cdsServicesVersion = MessageUtils.buffer().strong(getCdsServicesVersion());
		MessageBuilder cds4jVersion = MessageUtils.buffer().strong(getCds4jVersion());

		StringBuilder builder = new StringBuilder(2048);
		builder.append("\n").append(this.boundary).append("\n") //
				.append(JAVA_SDK).append(": ").append(cdsServicesVersion).append("\n") //
				.append(CDS4J).append(": ").append(cds4jVersion).append("\n") //
				.append(SPRING_BOOT).append(": ").append(getSpringBootVersion()).append("\n") //
				.append(MAVEN).append(": ").append(this.mavenVersion).append(", ").append(HOME).append(": ")
				.append(this.mavenHome).append("\n") //
				.append(JAVA).append(": ").append(this.javaVersion).append(", ").append(HOME).append(": ")
				.append(this.javaHome).append("\n") //
				.append(OS).append(": ").append(this.osName).append(", ").append(VERSION).append(": ")
				.append(this.osVersion).append(", ").append(ARCH).append(": ").append(this.osArch).append("\n");

		if (cdsVersion != null) {
			builder.append(cdsVersion);
		} else if (!this.excludeCds) {
			builder.append("Failed to execute 'cds version', no cds version information available.\n");
		}

		builder.append(this.boundary);
		logInfo(builder.toString());
	}

	private String getCdsServicesVersion() {
		Artifact artifact = this.finder.getArtifact(CDS_SERVICES_GROUPID, "cds-services-api");
		return artifact != null ? artifact.getVersion() : UNKNOWN;
	}

	private String getCds4jVersion() {
		Artifact artifact = this.finder.getArtifact(CDS_SERVICES_GROUPID, "cds4j-api");
		return artifact != null ? artifact.getVersion() : UNKNOWN;
	}

	private String getSpringBootVersion() {
		Artifact artifact = this.finder.getArtifact("org.springframework.boot", "spring-boot-starter");
		return artifact != null ? artifact.getVersion() : UNKNOWN;
	}

	private DependencyFinder scanDependencyGraph() throws MojoExecutionException {
		return DependencyFinder
				.scanDependencyGraph(
						findSrvProject(),
						this.depGraphBuilder,
						this.repositorySystemSession,
						(groupId, artifactId) ->
								Objects.equals(groupId, CDS_SERVICES_GROUPID) || Objects.equals(groupId, "org.springframework.boot"));
	}
}
