/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Enumeration;
import java.util.Locale;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.io.IOUtils;
import org.codehaus.plexus.util.FileUtils;

/**
 * This class extracts <code>*.zip</code> and <code>tar.gz</code> archives to a given destination directory.
 */
public class ArchiveUtils {

	private ArchiveUtils() {
		// avoid instances
	}

	/**
	 * Extracts an archive to a destination directory
	 *
	 * @param archiveFile          the archive to extract, either <code>*.zip</code> or <code>tar.gz</code>
	 * @param destinationDirectory the destination directory
	 * @throws FileNotFoundException if given archive doesn't exist
	 * @throws IOException           if an I/O error occurred
	 * @throws NullPointerException  if any of the parameters is <code>null</code>.
	 */
	public static void extract(File archiveFile, File destinationDirectory) throws IOException {
		requireNonNull(destinationDirectory, "destinationDirectory must not be null");

		String archiveName = requireNonNull(archiveFile, "archiveFile must not be null").getName()
				.toLowerCase(Locale.ENGLISH);
		String extension = FileUtils.extension(archiveName);

		try (FileInputStream fis = new FileInputStream(archiveFile)) {
			switch (extension) {
			case "zip":
			case "jar":
				extractZip(destinationDirectory, archiveFile);
				break;
			case "gz":
				archiveName = FileUtils.basename(archiveName, "." + extension);
				extension = FileUtils.extension(archiveName);
				if ("tar".equals(extension)) {
					extractTarGz(destinationDirectory, fis);
				} else {
					throw new IOException(format("Archives with extension %s not supported.", extension));
				}
				break;
			default:
				throw new IOException(format("Archives with extension %s not supported.", extension));
			}
		}
	}

	private static void checkTraversal(File destinationDir, File destPath) throws IOException {
		if (!destPath.getCanonicalPath().startsWith(destinationDir.getCanonicalPath())) {
			throw new IOException(format("Path traversal detected, extracting %s creates file outside of %s.", destPath,
					destinationDir));
		}
	}

	private static void extractTarGz(File destinationDir, FileInputStream fis) throws IOException {
		// TarArchiveInputStream can be constructed with a normal FileInputStream if we ever need to extract regular
		// *.tar files.
		try (TarArchiveInputStream tarIn = new TarArchiveInputStream(new GzipCompressorInputStream(fis))) {
			TarArchiveEntry tarEntry = tarIn.getNextTarEntry();

			while (tarEntry != null) {
				// create a file for this tarEntry
				File destPath = new File(destinationDir + File.separator + tarEntry.getName());

				// avoid: directory traversal”
				checkTraversal(destinationDir, destPath);

				// prepare destination file, for example, mkdirs
				Utils.prepareDestination(destPath, tarEntry.isDirectory());

				if (tarEntry.isFile()) {
					if (tarEntry.isLink()) {
						Path target = destPath.toPath().getParent().resolve(tarEntry.getLinkName()).normalize();
						Files.createLink(destPath.toPath(), target);
					} else if (tarEntry.isSymbolicLink()) {
						Path target = destPath.toPath().getParent().resolve(tarEntry.getLinkName()).normalize();
						Files.createSymbolicLink(destPath.toPath(), target);
					} else {
						if (!destPath.createNewFile()) {
							throw new IOException(format("Error creating file %s", destPath));
						}
						boolean isExecutable = (tarEntry.getMode() & 0100) > 0;
						if (!destPath.setExecutable(isExecutable)) {
							throw new IOException("Failed to set execution flag on " + destPath);
						}
						try (OutputStream out = new FileOutputStream(destPath)) {
							IOUtils.copy(tarIn, out);
						}
					}
				}

				tarEntry = tarIn.getNextTarEntry();
			}
		}
	}

	private static void extractZip(File destinationDir, File archiveFile) throws IOException {
		try (ZipFile zipFile = new ZipFile(archiveFile);) {
			Enumeration<? extends ZipEntry> entries = zipFile.entries();

			while (entries.hasMoreElements()) {
				ZipEntry zipEntry = entries.nextElement();

				// create a file for this *.zip entry
				File destPath = new File(destinationDir + File.separator + zipEntry.getName());

				// avoid: directory traversal
				checkTraversal(destinationDir, destPath);

				// prepare destination file, for example, mkdirs
				Utils.prepareDestination(destPath, zipEntry.isDirectory());

				if (!zipEntry.isDirectory()) {
					try (InputStream in = zipFile.getInputStream(zipEntry);
							OutputStream out = new FileOutputStream(destPath)) {
						IOUtils.copy(in, out);
					}
				}
			}
		}
	}
}