/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginExecution;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;

/**
 * Perform a CDS build with Java class generation executing the following goals of the
 * <strong>cds-maven-plugin</strong> in this order:
 * <ol>
 * <li><a href="install-node-mojo.html"><strong>cds:install-node</strong></a></li>
 * <li><a href="install-cdsdk-mojo.html"><strong>cds:install-cdsdk</strong></a></li>
 * <li><a href="cds-mojo.html"><strong>cds:cds</strong></a></li>
 * <li><a href="generate-mojo.html"><strong>cds:generate</strong></a></li>
 * </ol>
 * The list of goals and their order is fixed. Any Java property provided on the command line with the
 * <strong>-D</strong> option is forwarded to the invoked goals.<br>
 * Call {@code mvn cds:build} or {@code mvn com.sap.cds:cds-maven-plugin:build} to perform the CDS build. This goal is
 * very useful for local development to test changes to the CDS model.<br>
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 1.17.0
 */
@Mojo(name = "build", defaultPhase = LifecyclePhase.NONE, aggregator = true)
public class BuildMojo extends AbstractInvokerMojo {

	@Override
	public void execute() throws MojoExecutionException {
		ensureCliExecuted();

		executeBuild();
	}

	protected void executeBuild() throws MojoExecutionException {
		// get cds-maven-plugin from current pom.xml
		Plugin plugin = this.project.getModel().getBuild().getPluginsAsMap().get(PLUGIN_KEY);

		if (plugin != null) {
			executeGoal(plugin, "install-node");
			executeGoal(plugin, "install-cdsdk");
			executeGoal(plugin, "cds");
			executeGoal(plugin, "generate");
		} else {
			logInfo("Plugin %s not used in %s, skipping execution.", PLUGIN_KEY, this.project.getFile());
		}
	}

	protected void executeGoal(Plugin plugin, String goal) throws MojoExecutionException {
		PluginExecution pluginExecution = findGoalExecution(plugin, goal);
		if (pluginExecution != null) {
			executeMojo(plugin, goal, pluginExecution);
		} else {
			logDebug("Goal %s of plugin %s not used.", goal, plugin.getKey());
		}
	}
}
