/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.StringUtils;

import com.sap.cds.maven.plugin.util.Platform;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Execute an {@code npx} command on the CAP Java project.<br>
 *
 * @since 1.25.0
 */
@Mojo(name = "npx", defaultPhase = LifecyclePhase.INITIALIZE, aggregator = true, requiresProject = false)
public class NpxMojo extends AbstractNpmMojo {

	/**
	 * Add a line of space-separated arguments passed to {@code npx} for execution. For example: {@code cowsay goodbye!}.
	 */
	@Parameter(property = "cds.npx.arguments", required = true)
	private String arguments;

	/**
	 * Define an optional file to redirect standard and error output to. If not specified, the standard Maven logging
	 * is used.
	 */
	@Parameter(property = "cds.npx.outputFile")
	private File outputFile;

	/**
	 * Skip execution of this goal.
	 */
	@Parameter(property = "cds.npx.skip", defaultValue = "false")
	private boolean skip;

	/**
	 * The working directory to use during {@code npx} command execution. If not specified, the goal uses the directory
	 * containing a <strong>.cdsrc.json</strong> or <strong>package.json</strong> file. It goes up the project hierarchy
	 * on the file system until one of these files is found or the top-level project directory is reached.
	 */
	@Parameter(property = "cds.npx.workingDirectory")
	private File workingDirectory;

	/*
	 * Path to npx executable. internally used parameter, not configurable in pom.xml
	 */
	@Parameter(defaultValue = PARAM_NPX_EXECUTABLE, readonly = true)
	private File npxExec;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			executeNpx();
		} else {
			logInfo("Skipping execution.");
		}
	}

	/**
	 * Gets called to inject parameter "npxExec".
	 *
	 * @param npxExec a {@link File} pointing to npx executable
	 */
	public void setNpxExec(File npxExec) {
		if (npxExec != null && npxExec.canExecute()) {
			this.npxExec = npxExec;
			logInfo("Use npx provided by goal install-node: %s", strong(this.npxExec));
		}
	}

	/**
	 * Gets called to inject parameter "workingDirectory".
	 *
	 * @param workingDirectory a {@link File} pointing to working directory
	 */
	public void setWorkingDirectory(File workingDirectory) {
		if (workingDirectory != null) {
			if (workingDirectory.isDirectory()) {
				this.workingDirectory = workingDirectory;
				logInfo("Use configured working directory: %s", strong(this.workingDirectory));
			} else {
				logWarn("Configured working directory %s needs to be a valid directory. Please change it accordingly.",
						strong(workingDirectory));
			}
		}
	}

	private void executeNpx() throws MojoExecutionException {
		try {
			if (this.outputFile != null) {
				try (OutputStream fileOutput = new FileOutputStream(this.outputFile)) {
					execute(getWorkingDirectory(), getNpxExec(), fileOutput, null, null, getArguments());
				}
			} else {
				try (ByteArrayOutputStream fileOutput = new ByteArrayOutputStream()) {
					execute(getWorkingDirectory(), getNpxExec(), fileOutput, null, null, getArguments());
					logInfo(new String(fileOutput.toByteArray(), StandardCharsets.UTF_8));
				}
			}
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private String[] getArguments() throws MojoExecutionException {
		if (StringUtils.isBlank(this.arguments)) {
			throw new MojoExecutionException("No arguments provided.");
		}
		return Utils.splitByWhitespaces(this.arguments);
	}

	private File getNpxExec() throws IOException {
		if (this.npxExec == null) {
			this.npxExec = Utils.findExecutable(Platform.CURRENT.npx, this);
		}
		return this.npxExec;
	}

	private File getWorkingDirectory() {
		if (this.workingDirectory == null) {
			this.workingDirectory = findCdsWorkingDir();
		}
		return this.workingDirectory;
	}
}
