/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import static java.lang.String.format;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.maven.model.Plugin;
import org.apache.maven.model.PluginExecution;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.xml.Xpp3Dom;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.util.Semver;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Execute CDS commands on the CAP Java project.<br>
 * Call <code>mvn cds:cds</code> or <code>mvn com.sap.cds:cds-maven-plugin:cds</code> on the command line to execute all
 * configured CDS commands of the project in current directory.<br>
 * <br>
 * Several CDS commands can be configured in one execution block and they’re executed in the specified order. If a
 * command execution fails, the overall goal execution is stopped and the Maven build fails.<br>
 * <br>
 * <strong>Note:</strong> This goal requires the installation of the
 * <strong><a href="https://www.npmjs.com/package/@sap/cds-dk">@sap/cds-dk</a></strong> with a minimum version
 * <strong>4.0.0</strong>, either locally or globally. The goal
 * <strong><a href="install-cdsdk-mojo.html">cds:install-cdsdk</a></strong> can be used for this task.<br>
 * <br>
 *
 * @since 1.7.0
 */
@Mojo(name = "cds", defaultPhase = LifecyclePhase.GENERATE_SOURCES, aggregator = true)
public class CdsMojo extends AbstractCdsCliMojo {

	// @TODO: replace it with a minimum version provided by CDS4J
	private static final Semver MIN_CDSDK_VERSION = new Semver("4.0.0");

	static final String CDSDK_VERSION_KEY = "CDSDK_VERSION";

	/**
	 * Define a list of CDS commands to be executed in the specified order.<br>
	 * For example:
	 * <ul>
	 * <li><code>build --for java</code></li>
	 * <li><code>deploy --to h2 --with-mocks --dry > ${project.basedir}/src/main/resources/schema.sql</code></li>
	 * </ul>
	 */
	@Parameter
	private List<String> commands;

	/**
	 * Determine whether to generate JavaDoc for the generated Java POJOs.
	 *
	 * @since 1.17.0
	 */
	@Parameter(property = "cds.documentation", defaultValue = "true")
	private boolean documentation;

	/**
	 * Skip execution of this goal.
	 */
	@Parameter(property = "cds.cds.skip", defaultValue = "false")
	private boolean skip;

	@Override
	public void execute() throws MojoExecutionException {
		checkCdsdkVersion();

		if (isCliExecuted()) {
			executeViaCli();
		} else {
			executeViaLifecycle();
		}
	}

	@SuppressWarnings("unchecked")
	private void checkCdsdkVersion() throws MojoExecutionException {
		if (!this.skip) {
			Semver cdsdkSemver = (Semver) getPluginContext().get(CDSDK_VERSION_KEY);
			
			if (cdsdkSemver == null) { 
				String cdsVersionOutput = executeCdsVersion();
				cdsdkSemver = getCdsdkVersion(cdsVersionOutput);
				getPluginContext().put(CDSDK_VERSION_KEY, cdsdkSemver);
			}

			if (cdsdkSemver.compareTo(MIN_CDSDK_VERSION) < 0) {
				throw new MojoExecutionException(format(
						"Minimum required version %s of @sap/cds-dk not available, found version %s. Please update @sap/cds-dk to newer version.",
						MIN_CDSDK_VERSION, cdsdkSemver));
			}
		}
	}

	private void executeViaCli() throws MojoExecutionException {
		Plugin plugin = this.project.getModel().getBuild().getPluginsAsMap().get(PLUGIN_KEY);
		if (plugin != null) {
			PluginExecution pluginExec = findGoalExecution(plugin, "cds");

			if (pluginExec != null) {
				// get working directory from configuration or fallback to default value
				Xpp3Dom workingDirDom = ((Xpp3Dom) pluginExec.getConfiguration()).getChild("workingDirectory");
				File workingDir = workingDirDom != null ? new File(workingDirDom.getValue()) : getWorkingDirectory();

				Xpp3Dom commandsDom = ((Xpp3Dom) pluginExec.getConfiguration()).getChild("commands");

				for (Xpp3Dom command : commandsDom.getChildren()) {
					try {
						executeCds(workingDir, command.getValue(), null, getAdditionalCdsEnv());
					} catch (IOException e) {
						throw new MojoExecutionException(e.getMessage(), e);
					}
				}
			}
		}
	}

	private void executeViaLifecycle() throws MojoExecutionException {
		if (!this.skip) {
			File workDir = getWorkingDirectory();

			try {
				if (this.commands != null && !this.commands.isEmpty()) {
					for (String argLine : this.commands) {
						executeCds(workDir, argLine, null, getAdditionalCdsEnv());
					}

					// inform m2e about possible changes in resources directories, for example, edmx directory
					Utils.getResourceDirs(this.project).forEach(resourceDir -> super.buildContext.refresh(resourceDir));
				} else {
					logInfo("No commands configured, nothing to do.");
				}
				// catch everything as m2e doesn't show errors if a thrown exception
			} catch (Exception e) { // NOSONAR
				throw new MojoExecutionException(e.getMessage(), e);
			}
		} else {
			logInfo("Skipping execution.");
		}
	}

	@VisibleForTesting
	Semver getCdsdkVersion(String cdsVersionOutput) throws MojoExecutionException {
		Map<String, String> cdsdkVersions = Utils.parseCdsdkVersionOutput(cdsVersionOutput);

		// first look for version of locally installed @sap/cds-dk
		if (cdsdkVersions.containsKey("@sap/cds-dk")) {
			return new Semver(cdsdkVersions.get("@sap/cds-dk"));
		}

		// then look for version of globally installed @sap/cds-dk
		if (cdsdkVersions.containsKey("@sap/cds-dk (global)")) {
			return new Semver(cdsdkVersions.get("@sap/cds-dk (global)"));
		}

		throw new MojoExecutionException(
				format("Required @sap/cds-dk not installed, version information not available:%n%s", cdsVersionOutput));
	}

	private Map<String, String> getAdditionalCdsEnv() {
		// enable documentation generation if requested
		if (this.documentation) {
			return Collections.singletonMap("CDS_CDSC_DOCS", "true");
		}
		return null;
	}
}
