/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.sap.cds.maven.plugin.util.Utils;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.settings.Proxy;
import org.apache.maven.settings.Server;
import org.apache.maven.settings.crypto.DefaultSettingsDecryptionRequest;
import org.apache.maven.settings.crypto.SettingsDecrypter;
import org.apache.maven.shared.utils.logging.MessageUtils;
import org.codehaus.plexus.util.StringUtils;

/**
 * Download a Node.js distribution and install it on the local file system.<br>
 * If the requested version is already installed, the download and installation is skipped.<br>
 * <br>
 * This goal also considers proxy configurations in the <code>~/.m2/settings.xml</code>. If there's an active proxy for
 * the host in the {@code downloadUrl} configured, it is used.<br>
 * <br>
 * The goal also supports basic authentication at the {@code downloadUrl}. The optional parameter {@code serverId} can
 * be used to reference a server configuration from {@code settings.xml}, which provides username and password. See also
 * <a href="https://maven.apache.org/settings.html#servers">Settings Reference</a> for further details about server
 * configuration in {@code settings.xml}.<br>
 * <br>
 * Once executed, this goal exposes the following locations as project properties:
 * <ul>
 * <li><strong>${cds.node.executable}</strong>: Location of the <code>node</code> executable</li>
 * <li><strong>${cds.npm.executable}</strong>: Location of the <code>npm</code> executable</li>
 * <li><strong>${cds.npx.executable}</strong>: Location of the <code>npx</code> executable</li>
 * <li><strong>${cds.node.directory}</strong>: Directory containing the Node.js executables (node/npm/npx), can be used
 * to enhance the $PATH environment variable</li>
 * </ul>
 * <br>
 *
 * @since 1.7.0
 */
@Mojo(name = "install-node", defaultPhase = LifecyclePhase.INITIALIZE, aggregator = true, requiresProject = false)
public class InstallNodeMojo extends AbstractNodejsMojo {

	public static final String DEFAULT_NODE_VERSION = "v16.19.1";

	@Component(role = SettingsDecrypter.class)
	private SettingsDecrypter decrypter;

	/**
	 * Define the download URL of the Node.js distribution.
	 */
	@Parameter(property = "cds.install-node.downloadUrl", defaultValue = NodeInstaller.DEFAULT_NODEJS_DOWNLOAD_ROOT, required = true)
	private String downloadUrl;

	/**
	 * Force download and installation of Node.js, even if it's already installed.
	 */
	@Parameter(property = "cds.install-node.force", defaultValue = "false")
	private boolean force = false;

	/**
	 * Location of the target installation directory. If this parameter isn’t specified, the Node.js distribution is
	 * installed into a directory within the local Maven repository:
	 * <code>~/.m2/repository/com/sap/cds/cds-maven-plugin/cache/...</code>
	 */
	@Parameter(property = "cds.install-node.installDirectory")
	private File installDirectory;

	/**
	 * Define the Node.js version requested for installation.<br>
	 * <strong>IMPORTANT:</strong> The Node.js version has to start with 'v', for example 'v16.18.1'.
	 */
	@Parameter(property = "cds.install-node.nodeVersion", defaultValue = DEFAULT_NODE_VERSION, required = true)
	private String nodeVersion;

	/**
	 * Define a server identifier that points to a server configuration in {@code settings.xml} which provides the username and
	 * password used for basic authentication.
	 *
	 * @since 1.29.0
	 */
	@Parameter(property = "cds.install-node.serverId", required = false)
	private String serverId;

	/**
	 * Skip execution of this goal.
	 */
	@Parameter(property = "cds.install-node.skip", defaultValue = "false")
	private boolean skip;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			installNode();
		} else {
			logInfo("Skipping execution.");
		}
	}

	private List<Proxy> getProxyConfig() {
		if (this.settings.getProxies() == null || this.settings.getProxies().isEmpty()) {
			return Collections.emptyList();
		}

		List<Proxy> proxies = new ArrayList<>(this.settings.getProxies().size());

		for (Proxy mavenProxy : this.settings.getProxies()) {
			if (mavenProxy.isActive()) {
				DefaultSettingsDecryptionRequest decryptionRequest = new DefaultSettingsDecryptionRequest(mavenProxy);
				mavenProxy = this.decrypter.decrypt(decryptionRequest).getProxy();
				proxies.add(mavenProxy);
			}
		}
		return proxies;
	}

	/**
	 * Download and install Node.js if requested.
	 *
	 * @throws MojoExecutionException if Node.js download or installation failed.
	 */
	private void installNode() throws MojoExecutionException {
		if (!this.nodeVersion.startsWith("v")) {
			throw new MojoExecutionException(String.format(
					"Configured Node.js version \"%s\" is required to start with prefix 'v'", this.nodeVersion));
		}
		NodeCacheResolver nodeResolver = new NodeCacheResolver(this.repositorySystemSession);

		NodeInstaller installer = new NodeInstaller(this.nodeVersion, nodeResolver, this);
		installer.setProxies(getProxyConfig());
		installer.setForce(this.force);

		if (StringUtils.isNotBlank(this.serverId)) {
			logInfo("Using credentials from server %s for basic authentication.",
					MessageUtils.buffer().strong(this.serverId));
			Server server = Utils.decryptServer(this.serverId, this.session, this.decrypter, this);
			installer.setServer(server);
		}

		if (StringUtils.isNotBlank(this.downloadUrl)) {
			installer.setDownloadRoot(this.downloadUrl);
		}

		if (this.installDirectory != null) {
			installer.setInstallDirectory(this.installDirectory);
		}
		this.installDirectory = installer.install();

		// Node.js installation succeeded and the location properties can be set
		setProperty(PROP_NODE_EXECUTABLE, installer.getNodeExec());
		setProperty(PROP_NODE_DIR, installer.getNodeExec().getParentFile());
		setProperty(PROP_NPM_EXECUTABLE, installer.getNpmExec());
		setProperty(PROP_NPX_EXECUTABLE, installer.getNpxExec());

		logDebug("node install directory: " + this.installDirectory);
	}
}
