/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import org.codehaus.plexus.util.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * A utility class to manipulate pom.xml files.
 */
public class PomUtils {

	private static final String INDENT_1 = "\t";
	private static final String INDENT_2 = "\t\t";
	private static final String INDENT_3 = "\t\t\t";
	private static final String NEW_LINE = "\n";

	private PomUtils() {
		// to avoid instances
	}

	/**
	 * Adds a dependency to given pom.xml DOM if it doesn't exist yet.
	 *
	 * @param doc        the pom.xml DOM
	 * @param groupId    the dependency's groupId
	 * @param artifactId the dependency's artifactId
	 * @param version    the dependency's version
	 * @return <code>true</code> if dependency was added
	 */
	public static boolean addDependency(Document doc, String groupId, String artifactId, String version) {
		// look for <dependencies> element
		NodeList deps = doc.getElementsByTagName("dependencies");

		for (int i = 0; i < deps.getLength(); i++) {
			String parentNode = deps.item(i).getParentNode().getNodeName();

			// get global dependency list
			if ("project".equals(parentNode)) {
				Element depsElement = (Element) deps.item(i);

				// go one level deeper and get all child elements of type <dependency>
				NodeList dependencyNodes = depsElement.getChildNodes();

				if (!findDependency(dependencyNodes, groupId, artifactId)) {
					Node lastChild = depsElement.getLastChild();
					depsElement.insertBefore(doc.createTextNode(NEW_LINE), lastChild);
					depsElement.appendChild(doc.createTextNode(INDENT_1));
					Element depElement = createDependency(doc, groupId, artifactId, version);
					depsElement.appendChild(depElement);
					depsElement.appendChild(doc.createTextNode(NEW_LINE));
					depsElement.appendChild(doc.createTextNode(INDENT_1));
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Adds a module to given pom.xml DOM if it doesn't exist yet.
	 *
	 * @param doc    the pom.xml {@link Document} node
	 * @param module the module to add
	 * @return <code>true</code> if module was added
	 */
	public static boolean addModule(Document doc, String module) {
		// look for <modules> element
		NodeList modulesNodes = doc.getElementsByTagName("modules");

		if (modulesNodes.getLength() > 0) {
			// there’s only one modules element, get first
			Element modulesElement = (Element) modulesNodes.item(0);

			// go one level deeper and get all child elements of type <module>
			NodeList moduleList = modulesElement.getChildNodes();

			if (!findModule(moduleList, module)) {
				modulesElement.appendChild(doc.createTextNode(INDENT_1));
				Element moduleElement = createModule(doc, module);
				modulesElement.appendChild(moduleElement);
				modulesElement.appendChild(doc.createTextNode(NEW_LINE));
				modulesElement.appendChild(doc.createTextNode(INDENT_1));
				return true;
			}
		}
		return false;
	}

	private static Element createDependency(Document doc, String groupId, String artifactId, String version) {
		Element depElement = doc.createElement("dependency");
		depElement.appendChild(doc.createTextNode(NEW_LINE));

		depElement.appendChild(doc.createTextNode(INDENT_3));
		Element groupIdElement = doc.createElement("groupId");
		groupIdElement.setTextContent(groupId);
		depElement.appendChild(groupIdElement);
		depElement.appendChild(doc.createTextNode(NEW_LINE));

		depElement.appendChild(doc.createTextNode(INDENT_3));
		Element artifactIdElement = doc.createElement("artifactId");
		artifactIdElement.setTextContent(artifactId);
		depElement.appendChild(artifactIdElement);
		depElement.appendChild(doc.createTextNode(NEW_LINE));

		if (StringUtils.isNotBlank(version)) {
			depElement.appendChild(doc.createTextNode(INDENT_3));
			Element versionElement = doc.createElement("version");
			versionElement.setTextContent(version);
			depElement.appendChild(versionElement);
			depElement.appendChild(doc.createTextNode(NEW_LINE));
		}
		depElement.appendChild(doc.createTextNode(INDENT_2));

		return depElement;
	}

	private static Element createModule(Document doc, String module) {
		Element moduleElement = doc.createElement("module");
		moduleElement.appendChild(doc.createTextNode(INDENT_2));
		moduleElement.appendChild(doc.createTextNode(NEW_LINE));
		moduleElement.setTextContent(module);
		return moduleElement;
	}

	private static boolean findDependency(NodeList dependencyNodes, String groupId, String artifactId) {
		// loop over all dependencies
		for (int i = 0; i < dependencyNodes.getLength(); i++) {
			Node depNode = dependencyNodes.item(i);
			NodeList depChilds = depNode.getChildNodes();

			boolean isRequestedDependency = false;

			// loop over groupId, artifactId, ...
			for (int j = 0; j < depChilds.getLength(); j++) {
				Node depChild = depChilds.item(j);

				// there can be other type of nodes, for example, comments
				if (depChild.getNodeType() == Node.ELEMENT_NODE) {
					String name = depChild.getNodeName();
					switch (name) {
					case "groupId":
						String currentGroupId = ((Element) depChild).getTextContent();
						isRequestedDependency = groupId.equals(currentGroupId);
						break;
					case "artifactId":
						String currentArtifactId = ((Element) depChild).getTextContent();
						isRequestedDependency = artifactId.equals(currentArtifactId);
						break;
					default:
						break;
					}
				}
			}

			if (isRequestedDependency) {
				return true;
			}
		}
		return false;
	}

	private static boolean findModule(NodeList moduleNodes, String module) {
		// loop over all modules
		for (int i = 0; i < moduleNodes.getLength(); i++) {
			Node moduleNode = moduleNodes.item(i);

			// there can be other type of nodes, for example, comments
			if (moduleNode.getNodeType() == Node.ELEMENT_NODE && module.equals(moduleNode.getTextContent())) {
				return true;
			}
		}
		return false;
	}
}
