/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import java.io.File;

import javax.xml.transform.TransformerFactoryConfigurationError;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;

/**
 * Adds support of a feature to the CAP Java project. If the feature already exists, nothing is done.<br>
 * Call {@code mvn cds:add -Dfeature=MTXS} or {@code mvn com.sap.cds:cds-maven-plugin:add -Dfeature=MTXS} to add
 * Multitenancy support.<br>
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 2.1.0
 */
@Mojo(name = "add", aggregator = true)
public class AddMojo extends AbstractAddMojo {

	/**
	 * Defines the feature to add.<br>
	 * Allowed values are:
	 * <ul>
	 * <li><b>H2</b></li>H2 in-memory support is added
	 * <li><b>SQLITE</b></li>Sqlite in-memory support is added
	 * <li><b>MTX</b></li>Adds required Multitenancy dependencies and profiles to the CAP Java project, but not the MTXS
	 * sidecar.
	 * </ul>
	 */
	@Parameter(property = "feature", required = true)
	private String feature;

	/**
	 * Defines the profile name in application.yaml where this feature is added. An existing profile with the same name
	 * is overwritten by the new feature.
	 */
	@Parameter(property = "profile", required = true)
	private String profile;

	@Override
	public void execute() throws MojoExecutionException, MojoFailureException {
		ensureCliExecuted();

		MavenProject srvProject = findSrvProject();

		Addable featureImpl;
		try {
			switch (FeatureKey.valueOf(this.feature.toUpperCase())) {
			case H2:
				featureImpl = new AddableH2(srvProject, this);
				break;
			case SQLITE:
				File cdsrcJson = new File(super.getReactorBaseDirectory(), ".cdsrc.json");
				featureImpl = new AddableSqlite(srvProject, this, cdsrcJson);
				break;
			case MTX:
				featureImpl = new AddableMtx(srvProject, this);
				break;
			default:
				throw new MojoExecutionException(String.format("Feature %s not supported.", this.feature));
			}

			featureImpl.add(this.profile);
		} catch (TransformerFactoryConfigurationError | IllegalArgumentException e) {
			throw new MojoExecutionException(e);
		}
	}

	/**
	 * An enumeration with the keys of all supported features to add.
	 */
	public enum FeatureKey {
		/**
		 * Key for Sqlite support.
		 */
		SQLITE,

		/**
		 * Key for H2 support.
		 */
		H2,

		/**
		 * Key for Multitenancy support.
		 */
		MTX
	}
}
