/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import java.util.Arrays;
import java.util.List;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.CdsMojoLogger;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;

/**
 * Adds H2 database support to a CAP Java project. This includes the following changes:
 * <ul>
 * <li>add a H2 dependency to pom.xml</li>
 * <li>add a cds command line to generate schema-h2.sql to pom.xml</li>
 * <li>add the requested spring-boot profile to application.yaml</li>
 * </ul>
 */
class AddableH2 extends AbstractAddable {

	@VisibleForTesting
	static final String PROFILE = """
			---
			spring:
			  config.activate.on-profile: {0}
			  sql.init.schema-locations: classpath:schema-h2.sql
			cds:
			  datasource.auto-config.enabled: false
			  sql.supportedLocales: "*"
			""";

	@VisibleForTesting
	static final String CDS_COMMAND = "deploy --to h2 --dry > \"${project.basedir}/src/main/resources/schema-h2.sql\"";

	private static final List<String> TO_BE_REMOVED = Arrays.asList("spring.datasource.*", "spring.sql.*");

	AddableH2(MavenProject mvnProject, CdsMojoLogger logger) throws MojoExecutionException {
		super(mvnProject, logger);
	}

	@Override
	public void add(String profile) throws MojoExecutionException {
		// add a dependency to H2 and add required cds command into pom.xml
		boolean modified = addDependency("com.h2database", "h2", "runtime");
		modified |= addCdsCommand(CDS_COMMAND);

		if (modified) {
			savePomDocument();
		}

		// also add required spring-boot profile to applicaiton.yaml
		addProfileToApplicationYaml(getApplicationYaml(), profile, PROFILE, TO_BE_REMOVED);
	}
}
