/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import static com.sap.cds.maven.plugin.util.Utils.array;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.exec.ExecuteException;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.StringUtils;

import com.sap.cds.maven.plugin.util.Platform;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * An abstract base class for all Mojos using cds cli.
 */
abstract class AbstractCdsCliMojo extends AbstractNodejsMojo {

	/**
	 * The working directory to use during {@code cds} command execution. If not specified, the goal uses the directory
	 * containing a <strong>.cdsrc.json</strong> or <strong>package.json</strong> file. It goes up the project hierarchy
	 * on the file system until one of these files is found or the top-level project directory is reached.
	 */
	@Parameter
	private File workingDirectory;

	// internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${project.build.directory}/cds-build.log", readonly = true, required = true)
	private File cdsBuildOutputFile;

	// internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = PARAM_NPX_EXECUTABLE, readonly = true)
	private File npxExec;

	/**
	 * Gets called to inject parameter "npxExec".
	 *
	 * @param npxExec a {@link File} pointing to npx executable
	 */
	public void setNpxExec(File npxExec) {
		if (npxExec != null && npxExec.canExecute()) {
			this.npxExec = npxExec;
			logInfo("Using %s provided by goal install-node.", this.npxExec);
		}
	}

	/**
	 * Executes a cds command with given arguments in given working directory.
	 *
	 * @param workDir the working directory to use
	 * @param args    the cds command line arguments
	 * @param output  an optional {@link OutputStream}
	 * @throws ExecuteException if CDS execution failed with an error
	 * @throws IOException      if an I/O error occurred
	 */
	protected void executeCds(File workDir, String args, OutputStream output, Map<String, String> additionalEnv)
			throws IOException {
		// split and join to remove ugly whitespace characters like \n,\r,\t, and make command line more robust
		String argLine = StringUtils.join(Utils.splitByWhitespaces(args), " ");

		Utils.prepareDestination(this.cdsBuildOutputFile, false);
		Map<String, String> environment = additionalEnv != null ? new HashMap<>(additionalEnv) : new HashMap<>();
		environment.put("CDS_BUILD_OUTPUTFILE", this.cdsBuildOutputFile.getAbsolutePath());

		// set MSYSTEM and TERM on windows to empty values to solve issue with wrong environment in git-bash
		// see: https://github.com/npm/npx/issues/78#issuecomment-691102516
		if (Platform.CURRENT.isWindows()) {
			environment.put("MSYSTEM", "");
			environment.put("TERM", "");

			// double quotes need to be escaped on windows platform,
			// otherwise "npx -c cds ..." will fail on project paths containing a space
			argLine = argLine.replace("\"", "\\\"");
		}

		String[] allArgs = array("-c", "cds " + argLine);

		execute(workDir, getNpxExec(), output, environment, null, allArgs);
	}

	protected File getWorkingDirectory() {
		if (this.workingDirectory == null) {
			// determine cds working directory if not specified
			this.workingDirectory = findCdsWorkingDir();
		} else {
			logInfo("Using configured working directory: %s", this.workingDirectory);
		}
		return this.workingDirectory;
	}

	/**
	 * Executes command {@code cds version} and returns console output.
	 *
	 * @return the console output of the {@code cds version} command
	 * @throws MojoExecutionException if execution fails
	 */
	protected String executeCdsVersion() throws MojoExecutionException {
		try (ByteArrayOutputStream output = new ByteArrayOutputStream()) {
			// execute cds version and catch console output
			executeCds(getWorkingDirectory(), "version", output, null);

			String cdsVersionOutput = new String(output.toByteArray(), StandardCharsets.UTF_8);
			logDebug(cdsVersionOutput);

			return cdsVersionOutput;
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private File getNpxExec() throws IOException {
		if (this.npxExec == null) {
			this.npxExec = Utils.findExecutable(Platform.CURRENT.npx, this);
		}
		return this.npxExec;
	}
}
