/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import org.apache.commons.io.IOUtils;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.plugins.annotations.ResolutionScope;

import com.sap.cds.maven.plugin.AbstractCdsMojo;
import com.sap.cds.maven.plugin.util.Utils;

/**
 * Extracts models from dependency JARs and places them in {@code target/cds/} for reuse by the CDS Compiler.
 * <br>
 *
 * Models in JARs are expected to be placed in the {@code cds} resources folder. All contents in this folder are copied to {@code target/cds/}.
 * It is recommended to specify a unique folder structure for your models, for example by reusing group ID and artifact ID of the JAR.
 * <br>
 *
 * Here is an example JAR layout:
 *
 * <pre>
 * com/example/.../Some.class
 * cds/
 *   com.example.cds/
 *     my-reuse-models/
 *       index.cds
 *       Foo.cds
 *       data/
 *         com.example.cds-Foo.csv
 *       i18n/
 *         i18n.properties
 * </pre>
 *
 * You can then refer to these reuse models in your CDS files:
 *
 * <pre>
 * using { com.example.cds.Foo as Foo } from 'com.example.cds/my-reuse-models';
 * </pre>
 *
 * If your dependency is listed in the {@code srv/pom.xml} the reuse models are only available in CDS files placed in the {@code srv} folder.
 * This ensures that multiple independent application modules can define their reuse artifacts independently.
 * If you place a dependency in the root {@code pom.xml} the reuse models are available in all modules, including the {@code db} folder.
 *
 * @since 2.2.0
 */
@Mojo(name = "resolve", defaultPhase = LifecyclePhase.GENERATE_SOURCES, aggregator = true, requiresDependencyCollection = ResolutionScope.COMPILE_PLUS_RUNTIME, requiresDependencyResolution = ResolutionScope.COMPILE_PLUS_RUNTIME)
public class CdsResolveMojo extends AbstractCdsMojo {

	/**
	 * Skip extracting and resolving of CDS models.
	 */
	@Parameter(property = "cds.resolve.skip")
	private boolean skip;

	@Override
	public void execute() throws MojoExecutionException {
		if (!this.skip) {
			resolveModels();
		} else {
			logInfo("Skipping execution.");
		}
	}

	private void resolveModels() throws MojoExecutionException {
		Path targetPath = Paths.get(project.getBuild().getDirectory());
		for (Artifact artifact : project.getArtifacts()) {
			if (artifact.getFile().isFile() && "jar".equals(artifact.getType())) {
				try (JarFile jar = new JarFile(artifact.getFile())) {
					List<JarEntry> entries = jar.stream().filter(e -> !e.isDirectory() && e.getName().startsWith("cds/")).toList();
					if (!entries.isEmpty()) {
						logInfo("Extracting models from %s:%s:%s (%s)", artifact.getGroupId(), artifact.getArtifactId(), artifact.getVersion(), artifact.getFile().getAbsolutePath());
						for (JarEntry entry : entries) {
							File entryFile = targetPath.resolve(entry.getName()).toFile();
							logDebug("Writing %s", entryFile.getAbsolutePath());

							Utils.prepareDestination(entryFile, false);
							try (InputStream in = jar.getInputStream(entry);
								OutputStream out = new FileOutputStream(entryFile)) {
								IOUtils.copy(in, out);
							}
						}
					}
				} catch (IOException e) {
					throw new MojoExecutionException("Could not extract CDS models", e);
				}
			}
		}
	}
}
