package com.sap.cds.maven.plugin.util;

import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;

import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.config.YamlPropertiesFactoryBean;
import org.springframework.core.io.ByteArrayResource;
import org.yaml.snakeyaml.DumperOptions;
import org.yaml.snakeyaml.Yaml;

public class AppYamlUtils {

	// the boundary in an application.yaml between the different profiles as array of chars
	private static final char[] BOUNDARY = { '-', '-', '-', '\n' };

	// the boundary in an application.yaml between the different profiles as string
	private static final String BOUNDARY_STR = new String(BOUNDARY);

	private AppYamlUtils() {
		// avoid instances
	}

	/**
	 * Finds the boundaries of a document (spring profile) within an application.yaml.
	 * 
	 * @param appYaml    the whole application.yaml content
	 * @param startIndex the start position for the search, has to be inside the document for which the boundaries are
	 *                   identified
	 * @return an array with begin and end position of the document
	 */
	public static int[] findBoundaries(String appYaml, int startIndex) {
		int beginIndex = 0;

		// walk back to find beginning of profile
		for (int i = startIndex; i >= 0; i--) {
			if (appYaml.charAt(i) == BOUNDARY[0] && appYaml.charAt(i + 1) == BOUNDARY[1]
					&& appYaml.charAt(i + 2) == BOUNDARY[2] && appYaml.charAt(i + 3) == BOUNDARY[3]) {
				beginIndex = i + BOUNDARY.length;
				break;
			}
		}

		// find the end of the profile, either next --- or end of file
		int endIndex = appYaml.indexOf(BOUNDARY_STR, beginIndex + BOUNDARY.length);
		if (endIndex < 0) {
			endIndex = appYaml.length();
		}

		return new int[] { beginIndex, endIndex };
	}

	/**
	 * Returns the content and the position of a profile in an application.yaml.
	 * 
	 * @param appYaml the whole application.yaml content
	 * @param profile the profile name
	 * @return an {@link Optional} containing the profile content and position within in the application.yaml
	 */
	public static Optional<Pair<String, int[]>> findProfile(String appYaml, String profile) {
		int index = appYaml.indexOf("on-profile: " + profile);
		if (index > 0) {
			int[] boundaries = findBoundaries(appYaml, index);
			String profileContent = appYaml.substring(boundaries[0], boundaries[1]);
			return Optional.of(Pair.of(profileContent, boundaries));
		}
		return Optional.empty();
	}

	/**
	 * Merges an existing profile with a new profile. A merge includes the following steps:
	 * <ol>
	 * <li>Parse existing and new profile into {@link Properties properties}</li>
	 * <li>Remove conflicting properties (provided in the {@code toBeRemoved} list) from the existing properties</li>
	 * <li>Set new properties and overwrite existing ones</li>
	 * <li>Dump the properties in yaml format into a string</li>
	 * </ol>
	 * 
	 * @param existingProfile the existing profile
	 * @param newProfile      the new profile content
	 * @param toBeRemoved     a list with properties to be removed
	 * @return the enhanced profile content
	 */
	public static String mergeProfiles(String existingProfile, String newProfile, List<String> toBeRemoved) {

		// get existing profile as flattened properties
		Properties existingProfileProps = parseYamlProperties(existingProfile);

		// convert new profile content into flattened properties
		Properties newProfileProps = parseYamlProperties(newProfile);

		// first remove properties
		if (toBeRemoved != null) {
			toBeRemoved.forEach(path -> {
				// if path ends with ".*", all children are removed
				if (path.endsWith(".*")) {
					String prefix = path.substring(0, path.length() - 1);
					existingProfileProps.forEach((key, value) -> {
						if (((String) key).startsWith(prefix)) {
							existingProfileProps.remove(key);
						}
					});
				} else {
					existingProfileProps.remove(path);
				}
			});
		}

		// overwrite existing properties with new values
		newProfileProps.forEach(existingProfileProps::put);

		Map<String, Object> dataMap = new HashMap<>();
		existingProfileProps.forEach((key, value) -> {
			String[] pathSegments = key.toString().split("\\.");
			if (pathSegments.length > 0) {
				writeDeep(dataMap, pathSegments, value);
			}
		});

		DumperOptions options = new DumperOptions();
		options.setIndent(2);
		options.setPrettyFlow(true);
		options.setDefaultFlowStyle(DumperOptions.FlowStyle.BLOCK);

		return new Yaml(options).dump(dataMap);
	}

	public static String replaceProfile(String yamlContent, int[] boundaries, String newProfile) {
		// replace existing profile with the new one
		return yamlContent.substring(0, boundaries[0]) + newProfile
				+ yamlContent.subSequence(boundaries[1], yamlContent.length());
	}

	@SuppressWarnings("unchecked")
	public static void writeDeep(Map<String, Object> tree, String[] pathSegments, Object value) {
		if (pathSegments.length > 1) {
			// get or create next map
			Map<String, Object> object = (Map<String, Object>) tree.computeIfAbsent(pathSegments[0],
					key -> new HashMap<String, Object>());
			// truncate first path segment and go deeper
			writeDeep(object, Arrays.copyOfRange(pathSegments, 1, pathSegments.length), value);
		} else {
			// we are at the last path segment (leaf), put into current map
			tree.put(pathSegments[0], value);
		}
	}

	/**
	 * Parses the given profile into flattened {@link Properties properties}.
	 * 
	 * @param profile content of a single profile
	 * @return the flattened {@link Properties properties} of the profile
	 */
	public static Properties parseYamlProperties(String profile) {
		YamlPropertiesFactoryBean yamlPropsParser = new YamlPropertiesFactoryBean();
		yamlPropsParser.setResources(new ByteArrayResource(profile.getBytes(StandardCharsets.UTF_8)));
		return yamlPropsParser.getObject();
	}

}
