/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import static com.sap.cds.maven.plugin.util.Utils.array;
import static java.nio.file.LinkOption.NOFOLLOW_LINKS;
import static java.nio.file.StandardWatchEventKinds.ENTRY_CREATE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_DELETE;
import static java.nio.file.StandardWatchEventKinds.ENTRY_MODIFY;

import java.io.IOException;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import com.sap.cds.maven.plugin.CdsMojoLogger;

/**
 * Watches a directory, including the sub-tree, for changes on files. With lists of include and exclude GLOB pattern,
 * the watched files can be configured. Only if one of those files has changed, the event listener is notified.
 */
public class DirectoryWatcher {

	private final List<PathMatcher> includesMatcher;
	private final List<PathMatcher> excludesMatcher;
	private final CdsMojoLogger logger;
	private final Map<WatchKey, Path> watchKeys = new HashMap<>();
	private final FileChangedListener listener;
	private final Path root;
	private WatchService watcher;
	private volatile Thread watchThread; // NOSONAR

	/**
	 * Constructs a new {@link DirectoryWatcher} instance.
	 *
	 * @param root     the root of the directory sub-tree to watch
	 * @param listener the event listener
	 * @param logger   the mandatory logger
	 * @param includes an optional list with GLOB patterns describing the files to be include
	 * @param excludes an optional list with GLOB patterns describing the files to be exclude
	 * @throws NullPointerException if any of the required arguments is <code>null</code>
	 */
	public DirectoryWatcher(Path root, FileChangedListener listener, CdsMojoLogger logger, List<String> includes,
			List<String> excludes) {
		this.root = Objects.requireNonNull(root, "root");
		this.listener = Objects.requireNonNull(listener, "listener");
		this.logger = Objects.requireNonNull(logger, "logger");

		@SuppressWarnings("resource")
		FileSystem fs = FileSystems.getDefault();

		if (includes != null && !includes.isEmpty()) {
			this.includesMatcher = includes.stream().map(pattern -> fs.getPathMatcher("glob:" + pattern)).toList();
		} else {
			this.includesMatcher = Collections.singletonList(fs.getPathMatcher("glob:**"));
		}

		if (excludes != null && !excludes.isEmpty()) {
			this.excludesMatcher = excludes.stream().map(pattern -> fs.getPathMatcher("glob:" + pattern)).toList();
		} else {
			this.excludesMatcher = Collections.emptyList();
		}
	}

	/**
	 * Just for testing purpose.
	 *
	 * @return a Map containing all watch keys.
	 */
	Map<WatchKey, Path> getWatchKeys() {
		return this.watchKeys;
	}

	/**
	 * Starts asynchronously watching for changes on the file system.
	 *
	 * @throws IOException if registering for watching failed
	 */
	@SuppressWarnings("resource")
	public synchronized void start() throws IOException {
		if (this.watchThread == null) {
			this.watcher = FileSystems.getDefault().newWatchService();

			registerAll(this.root);

			this.watchThread = new Thread(() -> {
				processKeys();

				cleanup();
			});
			this.watchThread.start();
		}
	}

	/**
	 * Joins the watch thread and blocks until it's finished or interrupted.
	 *
	 * @throws InterruptedException if current thread is interrupted
	 */
	public synchronized void join() throws InterruptedException {
		if (this.watchThread != null && this.watchThread.isAlive()) {
			this.watchThread.join();
		}
	}

	/**
	 * Stops watching for changes on file system.
	 */
	public void stop() {
		if (this.watchThread != null) {
			this.watchThread.interrupt();
			this.watchThread = null;
		}
	}

	private void processKeys() {
		// process events endless until current thread gets interrupted
		while (this.watchThread != null && !this.watchThread.isInterrupted()) {
			WatchKey key = pollKey();

			if (key != null) {
				// check if directory is stilled watched
				Path dir = this.watchKeys.get(key);

				if (dir != null) {
					// consume all events for current key / dir pair
					pollEvents(key, dir);
				} else {
					removeWatchKey(key);
				}
			}
		}
	}

	/**
	 * Retrieves and returns the next watch key, waiting if necessary up to the specified wait time if none are yet
	 * present.
	 *
	 * @return a {@link WatchKey watch key} or <code>null</code> if it not available or interrupted.
	 */
	private WatchKey pollKey() {
		try {
			// wait for key to be signalled
			return this.watcher.poll(5, TimeUnit.SECONDS);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			this.logger.logDebug(e);
			return null;
		}
	}

	/**
	 * Polls and processes all events of given watch key.
	 *
	 * @param key the watch key
	 * @param dir the corresponding directory
	 */
	private void pollEvents(WatchKey key, Path dir) {
		boolean watchedFileChanged = false;

		for (WatchEvent<?> event : key.pollEvents()) {

			Path fileName = (Path) event.context();
			Path filePath = dir.resolve(fileName);

			// if directory is created, and watching recursively, then register it and its sub-directories
			if (event.kind() == ENTRY_CREATE && Files.isDirectory(filePath, NOFOLLOW_LINKS)) {
				try {
					registerAll(filePath);
				} catch (IOException x) {
					this.logger.logWarn(x);
				}
			}

			this.logger.logDebug("File changed: %s", filePath);

			// check next event / file if restart isn't required yet
			if (!watchedFileChanged && isIncluded(filePath)) {
				this.logger.logInfo("%s: %s", event.kind().name(), filePath);
				watchedFileChanged = true;
			}
		}

		resetWatchKey(key);

		this.listener.onChanged(watchedFileChanged);
	}

	private void cleanup() {
		// cancel all watch keys ...
		this.watchKeys.keySet().forEach(WatchKey::cancel);
		// ... and clear map
		this.watchKeys.clear();

		// finally close watch service
		try {
			this.watcher.close();
		} catch (IOException e) {
			this.logger.logWarn(e);
		} finally {
			this.watcher = null;
		}
	}

	private boolean isIncluded(Path filePath) {
		boolean included = this.includesMatcher.stream().anyMatch(matcher -> matcher.matches(filePath));

		if (included) {
			included = !isExcluded(filePath);
		}

		return included;
	}

	private boolean isExcluded(Path filePath) {
		return this.excludesMatcher.stream().anyMatch(matcher -> matcher.matches(filePath));
	}

	private void resetWatchKey(WatchKey key) {
		// reset key and remove from set if directory no longer accessible
		boolean valid = key.reset();
		if (!valid) {
			removeWatchKey(key);
		}
	}

	private void removeWatchKey(WatchKey key) {
		key.cancel();
		this.watchKeys.remove(key);
	}

	/**
	 * Register the given directory, and all its sub-directories, with the WatchService.
	 */
	private void registerAll(Path start) throws IOException {
		// ensure that directory still exists, otherwise a NoSuchFileException is thrown here
		if (Files.exists(start)) {

			// register directory and sub-directories
			Files.walkFileTree(start, new SimpleFileVisitor<Path>() {
				@Override
				public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) throws IOException {

					// ensure that directory still exists and is not excluded from watching
					if (!Files.exists(dir) || isExcluded(dir)) {
						DirectoryWatcher.this.logger.logDebug("Excluded \"%s\" from watching.", dir);
						return FileVisitResult.SKIP_SUBTREE;
					}

					WatchKey watchKey = dir.register(DirectoryWatcher.this.watcher,
							array(ENTRY_CREATE, ENTRY_DELETE, ENTRY_MODIFY),
							com.sun.nio.file.SensitivityWatchEventModifier.HIGH); // NOSONAR

					DirectoryWatcher.this.watchKeys.put(watchKey, dir);

					DirectoryWatcher.this.logger.logDebug("Included \"%s\" into watching.", dir);

					return FileVisitResult.CONTINUE;
				}
			});
		}
	}

	/**
	 * An interface used to listen on file changed events.
	 */
	@FunctionalInterface
	public interface FileChangedListener {

		/**
		 * Is called if any file in the project is changed. The parameter signals if
		 * a watched file is changed.
		 *
		 * @param watchedFileChanged true if a included file is changed. Otherwise false.
		 */
		void onChanged(boolean watchedFileChanged);
	}
}
