/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.StringTokenizer;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.CdsMojoLogger;

import org.apache.commons.io.IOUtils;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClients;
import org.apache.maven.settings.Proxy;
import org.apache.maven.settings.Server;
import org.codehaus.plexus.util.StringUtils;

/**
 * Used to download the Node archive from a given download URL.
 */
public class Downloader {

	// see: https://www.baeldung.com/httpclient-timeout#configure-timeouts-using-the-httpclient-43
	private static final RequestConfig requestCfg = RequestConfig.custom()
			// the time (ms) to establish the connection with the remote host
			.setConnectTimeout(10 * 1000)
			// the time (ms) waiting for data – after establishing the connection
			// maximum time of inactivity between two data packets
			.setSocketTimeout(60 * 1000).build();

	private final CdsMojoLogger logger;

	private final List<Proxy> proxies;

	private final Server server;

	/**
	 * Constructs a new {@link Downloader} instance.
	 *
	 * @param logger the required {@link CdsMojoLogger} instance.
	 * @throws NullPointerException if any of the required parameters is <code>null</code>.
	 */
	public Downloader(CdsMojoLogger logger) {
		this(null, null, logger);
	}

	/**
	 * Constructs a new {@link Downloader} instance.
	 *
	 * @param proxies an optional list of proxies
	 * @param server  an optional {@link Server} providing credentials for basic authentication.
	 * @param logger  the required {@link CdsMojoLogger} instance.
	 * @throws NullPointerException if any of the required parameters is <code>null</code>.
	 */
	public Downloader(List<Proxy> proxies, Server server, CdsMojoLogger logger) {
		this.proxies = proxies != null ? new ArrayList<>(proxies) : Collections.emptyList();
		this.server = server;
		this.logger = requireNonNull(logger, "logger must not be null");
	}

	/**
	 * Performs a download from given URL.
	 *
	 * @param downloadUrl the URL to download from
	 * @param destination the destination {@link File}
	 * @throws IOException          if download failed
	 * @throws NullPointerException if any of the required parameters is <code>null</code>.
	 */
	public void download(String downloadUrl, File destination) throws IOException {
		requireNonNull(destination, "destination must not be null");

		URI downloadUri = URI.create(requireNonNull(downloadUrl, "downloadUrl must not be null"));

		try (CloseableHttpResponse response = execute(downloadUri)) {
			int statusCode = response.getStatusLine().getStatusCode();
			if (statusCode != 200) {
				throw new IOException(format("Got error code %d from the server.", statusCode));
			}

			File destinationParent = destination.getParentFile();
			if (!destinationParent.exists() && !destinationParent.mkdirs())
				throw new IOException(format("Creating %s failed.", destinationParent));

			try (InputStream ris = response.getEntity().getContent();
					FileOutputStream fos = new FileOutputStream(destination)) {
				IOUtils.copy(ris, fos);
			}
		}
	}

	@VisibleForTesting
	Proxy getProxyForUrl(URI requestUri) {
		if (!this.proxies.isEmpty()) {
			for (Proxy proxy : this.proxies) {
				if (!isNonProxyHost(proxy, requestUri.getHost())) {
					return proxy;
				}
			}
			this.logger.logInfo("Could not find matching proxy for host: %s", requestUri.getHost());
		}
		return null;
	}

	private CloseableHttpResponse execute(URI requestUrl) throws IOException {
		// check if proxy is configured for the URL
		Proxy proxy = getProxyForUrl(requestUrl);
		if (proxy != null) {
			this.logger.logInfo("Downloading via proxy: %s:%d", proxy.getHost(), proxy.getPort());
			return executeViaProxy(proxy, requestUrl);
		}
		this.logger.logInfo("No proxy was configured, downloading directly.");
		return executeDirectly(requestUrl);
	}

	@SuppressWarnings("resource")
	private CloseableHttpResponse executeDirectly(URI requestUrl) throws IOException {

		CredentialsProvider credentialsProvider = null;
		if (this.server != null) {
			credentialsProvider = makeCredentialsProvider(requestUrl.getHost(), requestUrl.getPort(), this.server,
					null);
		}

		return buildHttpClient(credentialsProvider).execute(new HttpGet(requestUrl));
	}

	@SuppressWarnings("resource")
	private CloseableHttpResponse executeViaProxy(Proxy proxy, URI requestUri) throws IOException {
		CloseableHttpClient proxyClient;

		if (StringUtils.isNotEmpty(proxy.getUsername())) {
			CredentialsProvider credentialsProvider = makeCredentialsProvider(null, 0, this.server, proxy);
			proxyClient = buildHttpClient(credentialsProvider);
		} else {
			proxyClient = buildHttpClient(null);
		}

		HttpHost proxyHost = new HttpHost(proxy.getHost(), proxy.getPort(), proxy.getProtocol());
		RequestConfig requestConfig = RequestConfig.custom().setProxy(proxyHost).build();

		HttpGet request = new HttpGet(requestUri);
		request.setConfig(requestConfig);

		return proxyClient.execute(request);
	}

	// static helpers

	private static CloseableHttpClient buildHttpClient(CredentialsProvider credentialsProvider) {
		return HttpClients.custom().disableContentCompression().useSystemProperties()
				.setDefaultCredentialsProvider(credentialsProvider)
				// enable connection keep-alive if set in response header
				.setKeepAliveStrategy(DefaultConnectionKeepAliveStrategy.INSTANCE)
				// use request configuration with some configured timeouts
				.setDefaultRequestConfig(requestCfg).build();
	}

	private static boolean isNonProxyHost(Proxy proxy, String host) {
		if (StringUtils.isNotBlank(proxy.getNonProxyHosts())) {
			for (StringTokenizer tokenizer = new StringTokenizer(proxy.getNonProxyHosts(), "|"); tokenizer
					.hasMoreTokens();) {
				String pattern = tokenizer.nextToken().replace(".", "\\.").replace("*", ".*");
				if (host.matches(pattern)) {
					return true;
				}
			}
		}

		return false;
	}

	private static CredentialsProvider makeCredentialsProvider(String host, int port, Server server, Proxy proxy) {
		CredentialsProvider credentialsProvider = new BasicCredentialsProvider();

		if (server != null) {
			credentialsProvider.setCredentials(new AuthScope(host, port),
					new UsernamePasswordCredentials(server.getUsername(), server.getPassword()));
		}

		if (proxy != null) {
			credentialsProvider.setCredentials(new AuthScope(proxy.getHost(), proxy.getPort()),
					new UsernamePasswordCredentials(proxy.getUsername(), proxy.getPassword()));

		}
		return credentialsProvider;
	}
}