/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import static com.sap.cds.maven.plugin.AbstractCdsMojo.CDS_SERVICES_GROUPID;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Objects;

import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import javax.xml.xpath.XPathVariableResolver;

import com.google.common.annotations.VisibleForTesting;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.shared.utils.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * A utility class to manipulate pom.xml files.
 */
public class PomUtils {
	private static final String NS = "http://maven.apache.org/POM/4.0.0";
	private static final XPathFactory xPathFactory = XPathFactory.newInstance();
	private static final PomNamespaceResolver pomNsResolver = new PomNamespaceResolver(NS);
	private static final String GROUP_ID = "groupId";
	private static final String ARTIFACT_ID = "artifactId";
	private static final String NEW_LINE = "\n";
	private static final String INDENT = "\t";

	private PomUtils() {
		// to avoid instances
	}

	/**
	 * Adds a dependency to given pom.xml DOM if it doesn't exist yet.
	 *
	 * @param doc        the pom.xml DOM
	 * @param groupId    the dependency's groupId
	 * @param artifactId the dependency's artifactId
	 * @param version    the dependency's version
	 *
	 * @return <code>true</code> if dependency was added
	 * @throws MojoExecutionException if document manipulation failed
	 */
	public static boolean addDependency(Document doc, String groupId, String artifactId, String version, String scope)
			throws MojoExecutionException {
		try {
			if (getDependency(doc, groupId, artifactId) == null) {
				Element dependencies = getElement(doc, "/ns:project/ns:dependencies", null);
				if (dependencies != null) {
					Node lastChild = dependencies.getLastChild();
					String indent = getChildIndent(dependencies);
					dependencies.insertBefore(doc.createTextNode(NEW_LINE), lastChild);
					dependencies.insertBefore(doc.createTextNode(NEW_LINE), lastChild);
					dependencies.insertBefore(doc.createTextNode(indent), lastChild);
					dependencies.insertBefore(createDependency(doc, indent, groupId, artifactId, version, scope),
							lastChild);
					return true;
				}
			}
			return false;
		} catch (XPathExpressionException e) {
			throw new MojoExecutionException(e);
		}
	}

	/**
	 * Adds a module to given pom.xml DOM if it doesn't exist yet.
	 *
	 * @param doc    the pom.xml {@link Document} node
	 * @param module the module to add
	 *
	 * @return <code>true</code> if module was added
	 * @throws MojoExecutionException if document manipulation failed
	 */
	public static boolean addModule(Document doc, String module) throws MojoExecutionException {
		try {
			if (getModule(doc, module) == null) {
				Element modules = getElement(doc, "/ns:project/ns:modules", null);
				if (modules != null) {
					Node lastChild = modules.getLastChild();
					String indent = getChildIndent(modules);
					modules.insertBefore(doc.createTextNode(NEW_LINE), lastChild);
					modules.insertBefore(doc.createTextNode(indent), lastChild);
					modules.insertBefore(createModule(doc, module), lastChild);
					return true;
				}
			}
			return false;
		} catch (XPathExpressionException e) {
			throw new MojoExecutionException(e);
		}
	}

	/**
	 * Adds a CDS command line to the exeuction of the cds goal.
	 * 
	 * @param doc the pom.xml {@link Document} node
	 * @param cmd the CDS command line to add
	 *
	 * @return <code>true</code> if command was added
	 * @throws MojoExecutionException if document manipulation failed
	 */
	public static boolean addCdsCommand(Document doc, String cmd) throws MojoExecutionException {
		try {
			// try to find execution of goal cds
			Element execution = getPluginExecution(doc, CDS_SERVICES_GROUPID, "cds-maven-plugin", "cds");
			if (execution != null) {
				// try to find existing cds command with same command line
				Element command = getCdsCommand(doc, cmd);
				// if not found, the cds command is added to the configuration
				if (command == null) {
					NodeList commandsList = execution.getElementsByTagName("commands");
					if (commandsList.getLength() > 0) {
						Element commands = (Element) commandsList.item(0);
						Node lastChild = commands.getLastChild();
						String indent = getChildIndent(commands);
						commands.insertBefore(doc.createTextNode(NEW_LINE), lastChild);
						commands.insertBefore(doc.createTextNode(indent), lastChild);
						commands.insertBefore(createCdsCommand(doc, cmd), lastChild);
						return true;
					}
				}
			}
			return false;
		} catch (XPathExpressionException e) {
			throw new MojoExecutionException(e);
		}
	}

	@VisibleForTesting
	static Element getCdsCommand(Node node, String cdsCommand) throws XPathExpressionException {
		NodeList commands = getElements(node,
				"/ns:project/ns:build/ns:plugins/ns:plugin[ns:groupId='com.sap.cds' and ns:artifactId='cds-maven-plugin']/ns:executions/ns:execution[ns:goals/ns:goal/text()='cds']/ns:configuration/ns:commands/ns:command",
				null);
		if (commands != null)
			for (int i = 0; i < commands.getLength(); i++) {
				Element commandNode = (Element) commands.item(i);
				if (Objects.equals(commandNode.getTextContent(), cdsCommand)) {
					return commandNode;
				}
			}
		return null;
	}

	@VisibleForTesting
	static Element getDependency(Node node, String groupId, String artifactId) throws XPathExpressionException {
		return getElement(node,
				"/ns:project/ns:dependencies/ns:dependency[ns:groupId=$groupId and ns:artifactId=$artifactId]",
				new MapVariableResolver().with(GROUP_ID, groupId).with(ARTIFACT_ID, artifactId));
	}

	@VisibleForTesting
	static Element getPlugin(Node node, String groupId, String artifactId) throws XPathExpressionException {
		return getElement(node,
				"/ns:project/ns:build/ns:plugins/ns:plugin[ns:groupId=$groupId and ns:artifactId=$artifactId]",
				new MapVariableResolver().with(GROUP_ID, groupId).with(ARTIFACT_ID, artifactId));
	}

	@VisibleForTesting
	static Element getModule(Node node, String module) throws XPathExpressionException {
		return getElement(node, "/ns:project/ns:modules/ns:module[text()=$module]",
				new MapVariableResolver().with("module", module));
	}

	@VisibleForTesting
	static Element getPluginExecution(Node node, String groupId, String artifactId, String goal)
			throws XPathExpressionException {
		return getElement(node,
				"/ns:project/ns:build/ns:plugins/ns:plugin[ns:groupId=$groupId and ns:artifactId=$artifactId]/ns:executions/ns:execution[ns:goals/ns:goal/text()=$goal]",
				new MapVariableResolver().with(GROUP_ID, groupId).with(ARTIFACT_ID, artifactId).with("goal", goal));
	}

	private static Element getElement(Node node, String expression, XPathVariableResolver varResolver)
			throws XPathExpressionException {
		Object result = newXPath(varResolver).evaluate(expression, node, XPathConstants.NODE);
		if (result instanceof Element element) {
			return element;
		}
		return null;
	}

	private static NodeList getElements(Node node, String expression, XPathVariableResolver varResolver)
			throws XPathExpressionException {
		Object result = newXPath(varResolver).evaluate(expression, node, XPathConstants.NODESET);
		if (result instanceof NodeList nodes) {
			return nodes;
		}
		return null;
	}

	private static XPath newXPath(XPathVariableResolver varResolver) {
		XPath xPath = xPathFactory.newXPath();
		if (varResolver != null) {
			xPath.setXPathVariableResolver(varResolver);
		}
		xPath.setNamespaceContext(pomNsResolver);
		return xPath;
	}

	// DOM factory methods

	private static Element createCdsCommand(Document doc, String cdsCommand) {
		Element commandElement = doc.createElementNS(NS, "command");
		commandElement.appendChild(doc.createTextNode(cdsCommand));
		return commandElement;
	}

	private static Element createDependency(Document doc, String indent, String groupId, String artifactId,
			String version, String scope) {
		String indentChild = indent + INDENT;

		Element depElement = doc.createElementNS(NS, "dependency");
		depElement.appendChild(doc.createTextNode(NEW_LINE));

		depElement.appendChild(doc.createTextNode(indentChild));
		Element groupIdElement = doc.createElementNS(NS, GROUP_ID);
		groupIdElement.setTextContent(groupId);
		depElement.appendChild(groupIdElement);
		depElement.appendChild(doc.createTextNode(NEW_LINE));

		depElement.appendChild(doc.createTextNode(indentChild));
		Element artifactIdElement = doc.createElementNS(NS, ARTIFACT_ID);
		artifactIdElement.setTextContent(artifactId);
		depElement.appendChild(artifactIdElement);
		depElement.appendChild(doc.createTextNode(NEW_LINE));

		if (StringUtils.isNotBlank(version)) {
			depElement.appendChild(doc.createTextNode(indentChild));
			Element versionElement = doc.createElementNS(NS, "version");
			versionElement.setTextContent(version);
			depElement.appendChild(versionElement);
			depElement.appendChild(doc.createTextNode(NEW_LINE));
		}

		if (StringUtils.isNotBlank(scope)) {
			depElement.appendChild(doc.createTextNode(indentChild));
			Element versionElement = doc.createElementNS(NS, "scope");
			versionElement.setTextContent(scope);
			depElement.appendChild(versionElement);
			depElement.appendChild(doc.createTextNode(NEW_LINE));
		}
		depElement.appendChild(doc.createTextNode(indent));

		return depElement;
	}

	private static Element createModule(Document doc, String module) {
		Element moduleElement = doc.createElementNS(NS, "module");
		moduleElement.setTextContent(module);
		return moduleElement;
	}

	/**
	 * Returns the indent for children of given element.
	 * 
	 * @param element a DOM element
	 * @return an indent string
	 */
	private static String getChildIndent(Element element) {
		Node indentNode = element.getParentNode().getFirstChild();
		String indent = indentNode.getTextContent();
		indent = indent.replace("\n", "") + INDENT;
		return indent;
	}

	// helper classes

	/**
	 * A helper class to resolve variables in a XPath expression into a real value.
	 */
	static final class MapVariableResolver implements XPathVariableResolver {
		private final Map<String, Object> variables = new HashMap<>();

		MapVariableResolver with(String key, Object value) {
			this.variables.put(key, value);
			return this;
		}

		@Override
		public Object resolveVariable(QName variableName) {
			return this.variables.get(variableName.getLocalPart());
		}
	}

	/**
	 * A helper class to resolve namespace prefix "ns" into the pom.xml {@link PomUtils#NS namespace}.
	 */
	static final class PomNamespaceResolver implements NamespaceContext {
		private final String namespaceUri;

		PomNamespaceResolver(String namespaceUri) {
			this.namespaceUri = namespaceUri;

		}

		@Override
		public String getNamespaceURI(String prefix) {
			return prefix.equals("ns") ? this.namespaceUri : null;
		}

		@Override
		public Iterator<String> getPrefixes(String val) {
			return null;
		}

		@Override
		public String getPrefix(String uri) {
			return null;
		}
	}
}
