/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Component;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.DefaultProjectBuildingRequest;
import org.apache.maven.project.ProjectBuildingRequest;
import org.apache.maven.shared.dependency.graph.DependencyGraphBuilder;
import org.apache.maven.shared.dependency.graph.DependencyGraphBuilderException;
import org.apache.maven.shared.dependency.graph.DependencyNode;
import org.eclipse.aether.RepositorySystemSession;

import com.sap.cds.maven.plugin.AbstractCdsMojo;

/**
 * The base class for all CDS Mojos, which adds something to an existing CAP project.
 */
abstract class AbstractAddMojo extends AbstractCdsMojo {

	private static final String APPLICATION_JAVA = "Application.java";

	private String applicationPackage;

	@Component(role = DependencyGraphBuilder.class)
	private DependencyGraphBuilder depGraphBuilder;

	/**
	 * Defines settings and components that control the repository system.
	 */
	@Parameter(defaultValue = "${repositorySystemSession}", required = true, readonly = true)
	protected RepositorySystemSession repositorySystemSession;

	/**
	 * Finds the Java package containing the class Application.
	 *
	 * @return a Java package or <code>null</code> if not found
	 * @throws MojoExecutionException if <STRONG>Application.java</STRONG> couldn't be found
	 */
	String getApplicationPackage() throws MojoExecutionException {
		if (this.applicationPackage == null) {
			this.applicationPackage = findApplicationPackage();
		}
		return this.applicationPackage;
	}

	/**
	 * @return a {@link List} of string arrays containing path of class template and name of class to create.
	 */
	protected List<String[]> getIntegrationTestTemplates() {
		// get odata version, used to build template name
		return getODataVersions().stream().map(odataVersion -> {
			logDebug("Integration test template path: %s", odataVersion);

			// return array with template path and name of class to create
			return new String[] { "/CatalogServiceITest" + odataVersion + ".java",
					"CatalogServiceITest" + odataVersion };
		}).toList();
	}

	private Set<String> getODataVersions() {
		ProjectBuildingRequest request = new DefaultProjectBuildingRequest();
		request.setProject(findSrvProject());
		request.setRepositorySession(this.repositorySystemSession);

		Set<String> odataVersions = new HashSet<>();

		try {
			DependencyNode graph = this.depGraphBuilder.buildDependencyGraph(request, null);
			addODataVersions(graph.getChildren(), odataVersions);
		} catch (DependencyGraphBuilderException e) {
			logError(e);
		}

		return odataVersions;
	}

	private void addODataVersions(List<DependencyNode> dependencies, Set<String> odataVersions) {
		for (DependencyNode dependency : dependencies) {
			Artifact artifact = dependency.getArtifact();
			if (artifact.getArtifactId().equals("cds-adapter-odata-v2")) {
				odataVersions.add("V2");
			} else if (artifact.getArtifactId().equals("cds-adapter-odata-v4")) {
				odataVersions.add("V4");
			} else {
				addODataVersions(dependency.getChildren(), odataVersions);
			}
		}
	}

	/**
	 * Find the Java package of class Application in given Maven project.
	 *
	 * @param mvnProject
	 * @return the Java package or <code>null</code> if not found
	 * @throws MojoExecutionException if <STRONG>Application.java</STRONG> couldn't be found
	 */
	private String findApplicationPackage() throws MojoExecutionException {
		Path srcBaseDir = Paths.get(findSrvProject().getBuild().getSourceDirectory());

		// source base directory needs to be existing and a valid directory to avoid
		// IOExceptions
		if (Files.exists(srcBaseDir) && Files.isDirectory(srcBaseDir)) {

			// start walking on source base directory ...
			try (Stream<Path> walk = Files.walk(srcBaseDir)) {

				// ... and look for an Application.java file
				Optional<Path> result = walk
						.filter(file -> Files.isRegularFile(file) && file.endsWith(APPLICATION_JAVA)).findFirst();

				if (result.isPresent()) {
					// make path relative to base source directory ...
					String packagePath = srcBaseDir.relativize(result.get().getParent()).toString();
					// ... and convert it to a Java package
					return packagePath.replace(File.separatorChar, '.');
				}
			} catch (IOException e) {
				logError(e);
			}
		}

		throw new MojoExecutionException(String.format(
				"Couldn't find Application.java in source directory %s to determine Java package.", srcBaseDir));
	}
}
