/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import static com.sap.cds.maven.plugin.AbstractCdsMojo.CDS_SERVICES_GROUPID;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.text.MessageFormat;
import java.util.Optional;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactoryConfigurationError;

import com.sap.cds.maven.plugin.CdsMojoLogger;
import com.sap.cds.maven.plugin.util.AppYamlUtils;
import com.sap.cds.maven.plugin.util.PomUtils;
import com.sap.cds.maven.plugin.util.Utils;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * An abstract base class for all implementations of the interface {@link Addable}. It provides commonly used
 * functionality to add a new feature to a CAP Java project.
 */
abstract class AbstractAddable implements Addable {

	protected final CdsMojoLogger logger;
	protected final MavenProject project;
	private final Document pomDoc;

	protected AbstractAddable(MavenProject mvnProject, CdsMojoLogger logger) throws MojoExecutionException {
		this.project = mvnProject;
		this.logger = logger;
		try {
			this.pomDoc = Utils.parse(this.project.getFile());
		} catch (ParserConfigurationException | SAXException | IOException e) {
			throw new MojoExecutionException(e);
		}
	}

	/**
	 * Adds a dependency with the given values to the pom.xml of current project. If the dependency already exists,
	 * nothing is done.
	 * 
	 * @param groupId    the groupId of the maven dependency
	 * @param artifactId the artifactId of the maven dependency
	 * @param scope      the scope of the maven dependency, e.g. runtime or compile
	 * @return <code>true</code> if the dependency was added, otherwise <code>false</code>.
	 * @throws MojoExecutionException if an error occurred during adding dependency
	 */
	protected boolean addDependency(String groupId, String artifactId, String scope) throws MojoExecutionException {
		try {
			if (PomUtils.addDependency(this.pomDoc, groupId, artifactId, null, scope)) {
				this.logger.logInfo("Added dependency `%s:%s` to '%s'.", groupId, artifactId, this.project.getFile());
				return true;
			}
			this.logger.logWarn("Dependency '%s:%s' is already available, no changes have been made to pom.xml.",
					groupId, artifactId);
			return false;
		} catch (DOMException e) {
			throw new MojoExecutionException(e);
		}
	}

	/**
	 * Adds a new cds command to the execution section of the goal cds of the cds-maven-plugin.
	 * 
	 * @param cdsCommand the cds command to add, e.g. deploy --to sqlite
	 * @return <code>true</code> if the cds command was added, otherwise <code>false</code>.
	 * @throws MojoExecutionException if an error occurred during adding cds command
	 */
	protected boolean addCdsCommand(String cdsCommand) throws MojoExecutionException {
		try {
			if (PomUtils.addCdsCommand(this.pomDoc, cdsCommand)) {
				this.logger.logInfo("Added CDS command '%s' to cds-maven-plugin configuration in %s.", cdsCommand,
						this.project.getFile());
				return true;
			}
			this.logger.logWarn("CDS command '%s' not added to %s.", cdsCommand, this.project.getFile());
			return false;
		} catch (DOMException e) {
			throw new MojoExecutionException(e);
		}
	}

	protected boolean addCdsNpmInstallExecution() throws MojoExecutionException {
		boolean added = PomUtils.addPluginExecution(this.pomDoc, CDS_SERVICES_GROUPID, "cds-maven-plugin",
				"cds.npm-install", "npm", "<arguments>install</arguments>");
		if (added) {
			this.logger.logInfo("Added new execution of goal npm to cds-maven-plugin to perform.");
		} else {
			this.logger.logWarn("npm execution not added to cds-maven-plugin.");

		}
		// remove goal install-cdsdk, it's installed with a devDependency in package.json and npm install
		boolean removed = PomUtils.removePluginExecution(this.pomDoc, CDS_SERVICES_GROUPID, "cds-maven-plugin",
				"install-cdsdk");
		if (removed) {
			this.logger.logInfo("Removed execution of goal install-cdsdk from the cds-maven-plugin.");
		}
		return added || removed;
	}

	/**
	 * Searches for the application.yaml file in the resources directories of the Maven project.
	 * 
	 * @return the application.yaml
	 * @throws MojoExecutionException if application.yaml is not found
	 */
	protected File getApplicationYaml() throws MojoExecutionException {
		// look in resources directories for application.yaml, default location is "src/main/resources"
		return Utils.getResourceDirs(this.project).map(res -> new File(res, "application.yaml")).filter(File::exists)
				.findFirst().orElseThrow(
						() -> new MojoExecutionException("application.yaml not found in project's resources folder."));
	}

	/**
	 * Adds a new profile to the application.yaml. If the profile exists, it will be merged with the existing profile.
	 * 
	 * @param appYaml     the application.yaml file
	 * @param name        the profile name
	 * @param template    the profile content template
	 * @param toBeRemoved an array with properties to remove
	 * @throws MojoExecutionException if adding the profile failed
	 */
	protected void addProfileToApplicationYaml(File appYaml, String name, String template, String[] toBeRemoved)
			throws MojoExecutionException {
		// inject profile name into template
		String profileContent = MessageFormat.format(template, name);

		try {
			String appYamlContent = FileUtils.readFileToString(appYaml, StandardCharsets.UTF_8);

			// try to get profile from application.yaml
			Optional<Pair<String, int[]>> profileOpt = AppYamlUtils.findProfile(appYamlContent, name);

			// check if profile with given name already exists
			if (profileOpt.isPresent()) {
				Pair<String, int[]> existingProfile = profileOpt.get();

				// merge new profile into existing one
				profileContent = AppYamlUtils.mergeProfiles(existingProfile.getLeft(), profileContent, toBeRemoved);

				// then replace profile section in application.yaml
				appYamlContent = AppYamlUtils.replaceProfile(appYamlContent, existingProfile.getRight(),
						profileContent);
				this.logger.logWarn("Profile '%s' exists and has been updated.", name);
			} else {
				// add new profile at the end of the application.yaml
				if (appYamlContent.endsWith("\n")) {
					appYamlContent += profileContent;
				} else {
					appYamlContent += "\n" + profileContent;

				}
				this.logger.logInfo("Added profile '%s' to '%s'.", name, appYaml);
			}

			Files.writeString(appYaml.toPath(), appYamlContent, StandardOpenOption.WRITE,
					StandardOpenOption.TRUNCATE_EXISTING);
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	/**
	 * Saves the DOM of the pom.xml into the file.
	 * 
	 * @throws MojoExecutionException if saving failed
	 */
	protected void savePomDocument() throws MojoExecutionException {
		try {
			Utils.store(this.project.getFile(), this.pomDoc);
		} catch (TransformerFactoryConfigurationError | TransformerException | IOException e) {
			throw new MojoExecutionException(e);
		}
	}

}
