/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import java.io.File;
import java.io.IOException;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.CdsMojoLogger;
import com.sap.cds.maven.plugin.add.TemplateService.TemplateVariable;
import com.sap.cds.maven.plugin.util.Utils;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;

/**
 * Adds Liquibase support to a CAP Java project. This includes the following changes:
 * <ul>
 * <li>add a Liquibase dependency to pom.xml</li>
 * <li>add CDS commands to generate PostgreSQL specific changelog file</li>
 * <li>add a Liquibase master changelog file to {@code /srv/src/main/resources/db/changelog/db.changelog-master.yaml}
 * <li>create the folder src/main/resources/db/changelog/dev</li></li>
 * </ul>
 */
class AddableLiquibase extends AbstractAddable {

	@VisibleForTesting
	static final String DEV_PROFILE = """
			---
			spring:
			  liquibase:
			    contexts: "dev"
			""";

	/**
	 * Template for production ("cloud") profile.
	 */
	@VisibleForTesting
	static final String PRODUCTION_PROFILE = """
			---
			spring:
			  liquibase:
			    contexts: "!dev"
			""";

	private final TemplateService templateService = new TemplateService();

	private final String appPackage;

	AddableLiquibase(MavenProject mvnProject, String appPackage, CdsMojoLogger logger) throws MojoExecutionException {
		super(mvnProject, logger);
		this.appPackage = appPackage;
	}

	@Override
	public void add(String profile) throws MojoExecutionException {
		boolean modified = addDependency("org.liquibase", "liquibase-core", "runtime");
		modified |= addDependency("org.springframework.boot", "spring-boot-testcontainers", "test");
		modified |= addDependency("org.testcontainers", "postgresql", "test");
		modified |= addCdsCommand(
				"deploy --profile pg --dry > \"${project.basedir}/src/main/resources/db/changelog/dev/schema.sql\"");
		modified |= addCdsCommand(
				"deploy --model-only --dry > \"${project.basedir}/src/main/resources/db/changelog/dev/csn.json\"");

		if (modified) {
			savePomDocument();
		}

		String profileTemplate = profile.equalsIgnoreCase("cloud") ? PRODUCTION_PROFILE : DEV_PROFILE;

		// also add required spring-boot profile to applicaiton.yaml
		addProfileToApplicationYaml(getApplicationYaml(), profile, profileTemplate, null);

		TemplateVariable author = TemplateVariable.of("author", System.getProperty("user.name"));
		TemplateVariable dbms = TemplateVariable.of("dbms", "postgresql");
		try {
			// create empty folder src/main/resources/db/changelog/dev
			new File(this.project.getBasedir(), "src/main/resources/db/changelog/dev").mkdirs();

			// create master changelog file for liquibase
			this.templateService.createFileFromTemplate("/srv/src/main/resources/db/changelog/db.changelog-master.yaml",
					super.project.getBasedir().getParentFile(), true, author, dbms);

			File testSrcFolder = Utils.createDir(super.project.getBuild().getTestSourceDirectory());
			this.templateService.createClassFromTemplate(testSrcFolder, "/TestApplication.java", this.appPackage,
					"TestApplication", true);
			this.templateService.createClassFromTemplate(testSrcFolder, "/DatabaseConfiguration.java", this.appPackage,
					"DatabaseConfiguration", true);
		} catch (IOException e) {
			throw new MojoExecutionException(e);
		}

		this.logger.logInfo(
				"The Liquibase feature also adds support for TestContainers to the project, which requires a Docker installation. If you experience problems with TestContainers or don't have a Docker installation, remove this dependency.");
	}
}
