/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import java.io.File;
import java.io.IOException;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.CdsMojoLogger;
import com.sap.cds.maven.plugin.util.CdsrcUtils;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;

/**
 * Adds Sqlite support to a CAP Java project. This includes the following changes:
 * <ul>
 * <li>add a Sqlite dependency to pom.xml</li>
 * <li>add a cds command line to generate schema-sqlite.sql to pom.xml</li>
 * <li>add a new spring-boot profile to application.yaml</li>
 * <li>add a configuration to <b>.cdsrc.json</b></li>
 * </ul>
 */
class AddableSqlite extends AbstractAddable {

	@VisibleForTesting
	static final String PROFILE = """
			---
			spring:
			  config.activate.on-profile: {0}
			  sql.init:
			    mode: always
			    schema-locations: classpath:schema-sqlite.sql
			  datasource:
			    url: "jdbc:sqlite:file::memory:?cache=shared"
			    driver-class-name: org.sqlite.JDBC
			    hikari:
			      maximum-pool-size: 1
			      max-lifetime: 0
			cds:
			  sql.supportedLocales: ''*''
			""";

	@VisibleForTesting
	static final String CDS_COMMAND = "deploy --to sqlite --dry > \"${project.basedir}/src/main/resources/schema-sqlite.sql\"";

	private static final String[] TO_BE_REMOVED = { "spring.datasource.*", "spring.sql.*",
					"cds.datasource.auto-config.*" };

	private final File cdsrcJson;

	AddableSqlite(MavenProject mvnProject, CdsMojoLogger logger, File cdsrcJson) throws MojoExecutionException {
		super(mvnProject, logger);
		this.cdsrcJson = cdsrcJson;
	}

	@Override
	public void add(String profile) throws MojoExecutionException {
		// add dependency to sqlite and add required cds command to pom.xml
		boolean modified = addDependency("org.xerial", "sqlite-jdbc", "runtime");
		modified |= addCdsCommand(CDS_COMMAND);

		if (modified) {
			savePomDocument();
		}

		// also add required spring-boot profile to applicaiton.yaml
		addProfileToApplicationYaml(getApplicationYaml(), profile, PROFILE, TO_BE_REMOVED);

		try {
			CdsrcUtils utils = new CdsrcUtils(this.cdsrcJson);
			utils.enableBetterSqlite();
			utils.save();
		} catch (IOException e) {
			throw new MojoExecutionException(e);
		}

	}
}
