/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.util.Objects;

import org.eclipse.aether.RepositorySystemSession;
import org.eclipse.aether.artifact.DefaultArtifact;
import org.eclipse.aether.repository.LocalRepositoryManager;

import com.sap.cds.maven.plugin.util.Platform;

/**
 * A Node.js archive cache resolver using the local Maven repository as storage.
 */
public class NodeCacheResolver {

	private static final String GROUP_ID = "com.sap.cds.cds-maven-plugin.cache";

	private final LocalRepositoryManager localRepoManager;

	/**
	 * Constructs a new {@link NodeCacheResolver} instance.
	 *
	 * @param repositorySystemSession the active repository system session
	 * @throws NullPointerException if any of the required parameters is <code>null</code>.
	 */
	public NodeCacheResolver(RepositorySystemSession repositorySystemSession) {
		Objects.requireNonNull(repositorySystemSession, "repositorySystemSession");

		this.localRepoManager = repositorySystemSession.getLocalRepositoryManager();
		Objects.requireNonNull(this.localRepoManager, "localRepoManager");
	}

	/**
	 * Returns a Node file handle according the given version.
	 *
	 * @param version the required node version
	 * @return the resolved Node file handle
	 * @throws NullPointerException if any of the required parameters is <code>null</code>.
	 */
	public File resolve(String version) {
		Objects.requireNonNull(version, "version");

		return new File(this.localRepoManager.getRepository().getBasedir(),
				this.localRepoManager.getPathForLocalArtifact(createArtifact("archive", version)));
	}

	/**
	 * Returns a Node file handle according the given version.
	 *
	 * @param version the required Node version
	 * @return the resolved node file handle
	 * @throws NullPointerException if any of the required parameters is <code>null</code>.
	 */
	public File resolveUnpacked(String version) {
		Objects.requireNonNull(version, "version");

		return new File(this.localRepoManager.getRepository().getBasedir(),
				this.localRepoManager.getPathForLocalArtifact(createArtifact("unpacked", version)));
	}

	private static DefaultArtifact createArtifact(String artifactId, String version) {
		String newVersion = version.replaceAll("^v", "");

		return new DefaultArtifact(GROUP_ID, artifactId, Platform.CURRENT.nodeClassifier,
				Platform.CURRENT.archiveExtension, newVersion);
	}
}