/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.File;
import java.io.IOException;
import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Stream;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.util.DirectoryWatcher;
import com.sap.cds.maven.plugin.util.Utils;

import org.apache.commons.io.FileUtils;
import org.apache.maven.model.Plugin;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.LifecyclePhase;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.apache.maven.project.MavenProject;
import org.apache.maven.shared.invoker.InvocationRequest;
import org.apache.maven.shared.invoker.InvocationResult;
import org.apache.maven.shared.utils.cli.ShutdownHookUtils;

/**
 * Starts a CAP Java application and watches for changes in the CDS model. If changes are detected, a
 * <a href="build-mojo.html"><strong>cds:build</strong></a> is performed and the application is automatically restarted.
 *
 * Call <code>mvn cds:watch</code> or <code>mvn com.sap.cds:cds-maven-plugin:watch</code> to start the CAP Java
 * application.<br>
 * <br>
 * <strong>With Spring Boot Developer Tools</strong><br>
 * In order to get even faster development cycles locally, add an optional dependency to the Spring Boot Developer Tools
 * in the {@code pom.xml}:
 * 
 * <pre>
 * {@code
 * <dependency>
 *   <groupId>org.springframework.boot</groupId>
 *   <artifactId>spring-boot-devtools</artifactId>
 *   <optional>true</optional>
 * </dependency>}
 * </pre>
 *
 * In this case the watch goal leaves the restart of the application to the Developer Tools, which is faster. The CAP
 * Java application is started with a trigger file configured, used to signal the restart:
 * {@code --spring.devtools.restart.trigger-file=.reloadtrigger}. The watch goal touches this trigger file after the
 * <a href="build-mojo.html"><strong>cds:build</strong></a> is finished to signal the Spring Boot Developer Tools a safe
 * application restart.<br>
 * The trigger file is also touched, if any other file in the project has changed and <strong>no cds:build</strong> is
 * performed. This behaviour ensures that the application is restarted, if a Java source file was changed in an IDE and
 * a restart makes sense to test the changes.<br>
 * <br>
 * <strong>Without Spring Boot Developer Tools</strong><br>
 * Without a dependency to the Spring Boot Developer Tools, this goal recognises all changes of included files and fully
 * rebuilds and restarts the application.<br>
 * <br>
 * <strong>Start test application</strong><br>
 * By default, this goal uses {@code spring-boot:run} to start the application, but it also supports running an optional
 * test-application. This behaviour can be enabled by setting the property {@code -DtestRun} or {@code -DtestRun=true}
 * at the Maven command line.<br>
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 1.17.0
 */
@Mojo(name = "watch", defaultPhase = LifecyclePhase.NONE, aggregator = true)
public class WatchMojo extends BuildMojo {

	private static final String SPRING_BOOT_MAVEN_PLUGIN = "org.springframework.boot:spring-boot-maven-plugin";

	// TODO: make it configurable ?
	private static final String TRIGGER_FILE_NAME = ".reloadtrigger";

	/**
	 * A comma separated list with GLOB pattern describing directories and files to be excluded from watching for
	 * changes. Changes on these files are generally ignored. In case Spring Boot Developer tools are used, changes on
	 * all non-excluded files in the project directory trigger an update of the trigger file.<br>
	 * See <a href=
	 * "https://docs.oracle.com/javase/8/docs/api/java/nio/file/FileSystem.html#getPathMatcher-java.lang.String-">here</a>
	 * for further details about the supported GLOB patterns.
	 */
	@Parameter(property = "cds.watch.excludes", defaultValue = "**/node_modules,**/target,**/gen", required = true)
	private List<String> excludes;

	/**
	 * A comma separated list of GLOB patterns describing files to be included in watching for changes. Changes on these
	 * files will trigger a <strong>cds:build</strong>. In case Spring Boot Developer tools are not used, changes of
	 * these files will also trigger an explicit rebuild and restart of the application.<br>
	 * See <a href=
	 * "https://docs.oracle.com/javase/8/docs/api/java/nio/file/FileSystem.html#getPathMatcher-java.lang.String-">here</a>
	 * for further details about the supported GLOB patterns.
	 */
	@Parameter(property = "cds.watch.includes", defaultValue = "**/*.cds,**/*.csv", required = true)
	private List<String> includes;

	/**
	 * Indicates whether the CAP Java NG application will be started or not. If the application start is disabled, only
	 * a {@code cds:build} is executed on changes in the CDS model.
	 */
	private final boolean noStart;

	/**
	 * Indicates whether {@code spring-boot:run} or {@code spring-boot:test-run} is executed to start the application.
	 * If the property value is {@code true}, {@code spring-boot:test-run} is executed. Otherwise
	 * {@code spring-boot:run} is used.
	 * 
	 * @since 2.4.0
	 */
	@Parameter(property = "testRun", defaultValue = "false", required = true)
	private boolean testRun;
	
	// internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = "${project.build.outputDirectory}", readonly = true, required = true)
	private File outputDir;

	private Process mvnProcess;

	private DirectoryWatcher watcher;

	private boolean useDeveloperTools;

	private Plugin resourcesPlugin;

	private Optional<File> triggerFile;

	// used for testing purpose
	private volatile Instant lastRestart;
	private volatile boolean simulate = false;

	private MavenProject srvProject;

	public WatchMojo() {
		this.noStart = false;
	}

	protected WatchMojo(boolean noStart) {
		this.noStart = noStart;
	}

	@Override
	public void execute() throws MojoExecutionException {
		// ensure this goal is executed from command line.
		ensureCliExecuted();

		initialize();

		if (!this.simulate) {
			if (this.noStart) {
				// initial cds build
				performCdsBuild();
			} else {
				// initial start of the application
				startApplication();
			}
		}

		this.lastRestart = Instant.now();

		try {
			this.watcher.start();

			this.watcher.join();
		} catch (IOException e) {
			logError(e);
			throw new MojoExecutionException(e.getMessage(), e);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			logDebug(e);
			// as thread was interrupted also stop running application
			stopApplication();
		}
	}

	@VisibleForTesting
	void setSimulate(boolean simulate) {
		this.simulate = simulate;
	}

	@VisibleForTesting
	synchronized Instant getLastRestart() {
		return this.lastRestart;
	}

	/**
	 * Gets called when a file has changed.
	 */
	private synchronized void onFileChanged(boolean watchedFileChanged) {
		if (this.useDeveloperTools || this.noStart) {
			// just perform a cds build, restart will be done by developer tools
			if (watchedFileChanged) {
				if (!this.simulate) {
					// if the initial application start failed, the mvn process needs to be started again
					if (this.mvnProcess != null && !this.mvnProcess.isAlive() && !this.noStart) {
						this.logInfo("The Maven process exited unexpectedly and will be restarted.");
						try {
							startApplication();
						} catch (MojoExecutionException e) {
							logError(e);
						}
					} else {
						performCdsBuild();
					}
				}
				this.lastRestart = Instant.now();
			}

			// also touch trigger file for other resources, e.g. java files ->
			// dev. tools decide whether to restart or not
			touchTriggerFileIfPresent();
		} else {
			if (watchedFileChanged) {
				if (!this.simulate) {
					// first stop a running application instance
					stopApplication();

					try {
						startApplication();
					} catch (MojoExecutionException e) {
						logError(e);
					}
				}
				this.lastRestart = Instant.now();
			}
		}
	}

	private void touchTriggerFileIfPresent() {
		if (this.triggerFile.isPresent()) {
			File file = this.triggerFile.get();
			// watch: always touch trigger file -> will be created if missing
			// auto-build: only touch trigger file if existing
			if (!this.noStart /* watch */ || file.exists() /* auto-build */) {
				try {
					FileUtils.touch(file);
					logDebug("Touched Spring Boot devtools trigger file: %s", file);
				} catch (IOException e) {
					throw new RuntimeException( // NOSONAR
							String.format("Failed to touch devtools trigger file: %s", file), e);
				}
			}
		}
	}

	private void performCdsBuild() {
		try {
			// first perform a cds:build ...
			super.executeBuild();

			logDebug("Copying resources with org.apache.maven.plugins:maven-resources-plugin.");

			// ... then copy build result to target/classes folder to make it visible to dev. tools -> triggers restart
			executeGoal(this.resourcesPlugin, "resources");
		} catch (MojoExecutionException e) {
			logWarn(e);
		}
	}

	private void startApplication() throws MojoExecutionException {
		InvocationRequest request = createRequest(getStartGoals(), "spring-boot.run.arguments",
				"--spring.devtools.restart.trigger-file=" + TRIGGER_FILE_NAME);

		logInfo("Starting application %s", super.project.getName());

		this.mvnProcess = executeMaven(request);
	}

	/**
	 * Stops the spring-boot application.
	 *
	 * @return the application stop invocation {@link InvocationResult result}
	 */
	private void stopApplication() {
		if (this.mvnProcess != null) {
			logInfo("Stopping application %s", super.project.getName());

			ProcessHandle procHandle = this.mvnProcess.toHandle();
			// destroy all sub-processes
			procHandle.descendants().forEach(handle -> {
				if (handle.destroy()) {
					logDebug("Successfully requested destroying of sub-process with pid '%d'", handle.pid());
				} else {
					logError("Failed to request destroying of sub-process with pid '%d'", handle.pid());
				}
			});
			// finally destroy maven process itself
			if (procHandle.destroy()) {
				logDebug("Successfully requested destroying of maven process with pid '%d'", procHandle.pid());
			} else {
				logError("Failed to request destroying of maven process with pid '%d'", procHandle.pid());
			}

			try {
				boolean exited = this.mvnProcess.waitFor(5, TimeUnit.SECONDS);
				if (exited) {
					logInfo("Stopped application %s: rc=%d", super.project.getName(), this.mvnProcess.exitValue());
				} else {
					this.mvnProcess.destroyForcibly();
					logInfo("Destroyed application %s", super.project.getName());
				}
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				logError(e);
			} finally {
				this.mvnProcess = null;
			}
		}
	}

	private void initialize() throws MojoExecutionException {
		this.srvProject = findSrvProject();

		// if no-start is declared we don't need the spring boot plugin
		if (!this.noStart) {
			// ensure the spring-boot maven plugin is available
			ensurePlugin(SPRING_BOOT_MAVEN_PLUGIN);

			// search for developer tools dependency in current pom.xml
			this.useDeveloperTools = hasDependency("org.springframework.boot:spring-boot-devtools:jar");

			if (this.useDeveloperTools) {
				logInfo("Found dependency to Spring-Boot Developer Tools, will use it for a faster restart of the CAP Java application.");
			} else {
				logInfo("No Spring-Boot Developer Tools found.");
			}
		}

		// lookup reload trigger file in first available (test) resources directory,
		// usually src/main/resources or src/test/resources
		Stream<File> stream = this.testRun ? Utils.getTestResourceDirs(this.srvProject)
				: Utils.getResourceDirs(this.srvProject);
		this.triggerFile = stream.findFirst().map(resourceDir -> new File(resourceDir, TRIGGER_FILE_NAME));

		// get maven-resources-plugin for later usage after the cds:build is performed
		this.resourcesPlugin = ensurePlugin("org.apache.maven.plugins:maven-resources-plugin");

		ShutdownHookUtils.addShutDownHook(new Thread(this::stopApplication));

		this.watcher = new DirectoryWatcher(getReactorBaseDirectory().toPath(), this::onFileChanged, this,
				this.includes, this.excludes);
	}

	/**
	 * Ensures that current project uses the plugin with the given key.
	 *
	 * @param pluginKey the plugin key
	 * @return the {@link Plugin} for given key
	 * @throws MojoExecutionException if project doesn't use plugin with given key
	 */
	private Plugin ensurePlugin(String pluginKey) throws MojoExecutionException {
		Plugin plugin = getPlugin(pluginKey);
		if (plugin == null) {
			throw new MojoExecutionException(
					"Project doesn't use plugin " + pluginKey + ". To watch this project, it's required.");
		}
		return plugin;
	}

	private Plugin getPlugin(String pluginKey) {
		return this.srvProject.getModel().getBuild().getPluginsAsMap().get(pluginKey);
	}

	private boolean hasDependency(String depKey) {
		return this.srvProject.getDependencies().stream()
				.anyMatch(dependency -> dependency.getManagementKey().equals(depKey));
	}

	private List<String> getStartGoals() {
		return Collections.singletonList(SPRING_BOOT_MAVEN_PLUGIN + ":" + (this.testRun ? "test-run" : "run"));
	}
}
