/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import java.io.File;
import java.io.IOException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import com.sap.cds.maven.plugin.add.TemplateService.TemplateVariable;
import com.sap.cds.maven.plugin.util.PomUtils;
import com.sap.cds.maven.plugin.util.Utils;

import org.apache.maven.model.Model;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * Adds an integration test module to the CAP Java project. The module is created in the
 * <strong>./integration-tests</strong> folder and an entry to the list of modules in the parent pom.xml is added.<br>
 * Call {@code mvn cds:addIntegrationTest} or {@code mvn com.sap.cds:cds-maven-plugin:addIntegrationTest} to add an
 * integration test module.<br>
 * It automatically detects the OData version of the CAP Java project and adds the corresponding integration test
 * class.<br>
 * <br>
 * <strong>Note:</strong> This goal can only be executed from the command line.<br>
 * <br>
 *
 * @since 1.5.0
 */
@Mojo(name = "addIntegrationTest", aggregator = true)
public class AddIntegrationTestMojo extends AbstractAddMojo {
	static final String ITEST_MODULE = "integration-tests";

	static final String IT_PACKAGE = ".it";

	private boolean createTestClass = false;

	/**
	 * Define if existing files are overwritten ({@code true}) without warning.
	 */
	@Parameter(property = "overwrite", defaultValue = "false")
	private boolean overwrite;

	private final TemplateService templateService = new TemplateService();

	@Override
	public void execute() throws MojoExecutionException {
		ensureCliExecuted();

		File cdsFile = new File(findSrvProject().getBasedir(), AddSampleMojo.CAT_SERVICE_CDS);
		this.createTestClass = cdsFile.exists();

		try {
			addIntegrationTest();

			updateParentPom();
		} catch (IOException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private void addIntegrationTest() throws IOException, MojoExecutionException {
		logDebug("overwrite: %s", this.overwrite);

		if (this.createTestClass) {
			String applicationPackage = getApplicationPackage();
			String packageName = applicationPackage + IT_PACKAGE;

			logDebug("application package: %s", applicationPackage);
			logDebug("integration test package: %s", packageName);
			TemplateVariable applicationPackagePair = TemplateVariable.of("applicationPackage", applicationPackage);

			File testSourceFolder = new File(this.project.getBasedir(), ITEST_MODULE + "/src/test/java");
			logDebug("test source folder: %s", testSourceFolder);

			// create integration test classes for each configured OData version
			getIntegrationTestTemplates().forEach(template -> {
				try {
					this.templateService.createClassFromTemplate(testSourceFolder, template[0], packageName,
							template[1], this.overwrite, applicationPackagePair);
				} catch (IOException e) {
					logWarn(e);
				}
			});
		} else {
			logWarn("Integration test class not added, sample CDS model is missing. You can add it by executing goal addSample afterwards.");
		}

		// add pom.xml to newly created integration test module
		addItPom();
	}

	private void addItPom() throws IOException {
		Model model = this.project.getModel();
		TemplateVariable parentGroupId = TemplateVariable.of("parentGroupId", model.getGroupId());
		TemplateVariable parentArtifactId = TemplateVariable.of("parentArtifactId", model.getArtifactId());

		// create pom.xml
		Model srvModel = findSrvProject().getModel();
		TemplateVariable artifactId = TemplateVariable.of("artifactId", srvModel.getArtifactId());
		this.templateService.createFileFromTemplate("/" + ITEST_MODULE + "/pom.xml", this.project.getBasedir(),
				this.overwrite, parentGroupId, parentArtifactId, artifactId);
	}

	private void updateParentPom() throws MojoExecutionException {
		try {
			Document doc = Utils.parse(this.project.getFile());

			boolean added = PomUtils.addModule(doc, ITEST_MODULE);
			if (added) {
				Utils.store(this.project.getFile(), doc);
			}
		} catch (ParserConfigurationException | SAXException | IOException | TransformerException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}
}
