/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import static com.sap.cds.maven.plugin.AbstractCdsMojo.CDS_SERVICES_GROUPID;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.CdsMojoLogger;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;

/**
 * Adds Multitenancy support to a CAP Java project. This includes the following changes:
 * <ul>
 * <li>add a dependency to <b>pom.xml</b></li>
 * <li>add a new spring-boot profile to <b>application.yaml</b></li>
 * </ul>
 */
class AddableMtx extends AbstractAddable {

	/**
	 * Template for any other profile except production.
	 */
	@VisibleForTesting
	static final String ANY_OTHER_PROFILE = """
			---
			spring:
			  config.activate.on-profile: {0}
			cds:
			  multi-tenancy:
			    mtxs.enabled: true
			    sidecar.url: http://localhost:4005/ # in production mode overwrite with the URL from mta.yaml
			""";

	/**
	 * Template for production ("cloud") profile.
	 */
	@VisibleForTesting
	static final String PRODUCTION_PROFILE = """
			---
			spring:
			  config.activate.on-profile: cloud
			cds:
			  multi-tenancy:
			    mtxs.enabled: true
			""";

	AddableMtx(MavenProject mvnProject, CdsMojoLogger logger) throws MojoExecutionException {
		super(mvnProject, logger);
	}

	@Override
	public void add(String profile) throws MojoExecutionException {
		boolean production = profile.equalsIgnoreCase("cloud");

		// add dependency to mtx feature to pom.xml
		boolean modified = addDependency(CDS_SERVICES_GROUPID, "cds-feature-mt", "runtime");

		if (!production) {
			// for local testing add a dependency to sqlite
			modified |= addDependency("org.xerial", "sqlite-jdbc", "runtime");
		}

		if (modified) {
			savePomDocument();
		}

		String profileTemplate = production ? PRODUCTION_PROFILE : ANY_OTHER_PROFILE;

		// also add spring-boot profile to applicaiton.yaml
		addProfileToApplicationYaml(getApplicationYaml(), profile, profileTemplate, null);
	}
}
