/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import com.google.common.annotations.VisibleForTesting;
import com.sap.cds.maven.plugin.CdsMojoLogger;

import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;

/**
 * Adds PostgreSQL database support to a CAP Java project. This includes the following changes:
 * <ul>
 * <li>add a PostgreSQL dependency to pom.xml</li>
 * <li>add the required spring-boot profile to application.yaml</li>
 * </ul>
 */
class AddablePostgreSQL extends AbstractAddable {

	@VisibleForTesting
	static final String PROD_PROFILE = """
			---
			spring:
			  config:
			    activate:
			      on-profile: {0}
			  datasource:
			    embedded-database-connection: none
			    driver-class-name: org.postgresql.Driver
			""";

	static final String DEV_PROFILE = """
			---
			spring:
			  config:
			    activate:
			      on-profile: {0}
			  datasource:
			    embedded-database-connection: none
			    driver-class-name: org.postgresql.Driver
			    url: jdbc:postgresql://localhost:5432/postgres
			    username: postgres
			    password: postgres
			cds:
			  datasource:
			    auto-config:
			      enabled: false
			""";

	private static final String[] TO_BE_REMOVED = { "spring.datasource.*", "spring.sql.*", "cds.sql.supportedLocales" };

	AddablePostgreSQL(MavenProject mvnProject, CdsMojoLogger logger) throws MojoExecutionException {
		super(mvnProject, logger);
	}

	@Override
	public void add(String profile) throws MojoExecutionException {
		String profileTemplate = profile.equalsIgnoreCase("cloud") ? PROD_PROFILE : DEV_PROFILE;

		boolean modified = addDependency("com.sap.cds", "cds-feature-postgresql", "runtime");
		modified |= addCdsNpmInstallExecution();

		if (modified) {
			savePomDocument();
		}

		// also add required spring-boot profile to applicaiton.yaml
		addProfileToApplicationYaml(getApplicationYaml(), profile, profileTemplate, TO_BE_REMOVED);

		this.logger.logInfo("Please consider using PostgreSQL as your primary database by removing in-memory database from your configuration.");
	}
}
