/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.add;

import static java.lang.String.format;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;

import org.apache.commons.io.IOUtils;

import com.sap.cds.maven.plugin.util.Utils;

/**
 * This service creates files and Java classes from templates.
 */
public class TemplateService {

	/**
	 * Creates a new Java class source file from a specified resource template.
	 *
	 * @param javaSrcFolder target folder of the new Java source file
	 * @param templatePath  resource path of the template to use
	 * @param packageName   package of the Java class
	 * @param className     name of the Java class
	 * @param overwrite     use <code>true</code> if existing Java class shall be overwritten
	 * @param vars          additional array of key/value pairs used to replace variables in template
	 * @throws IOException if an I/O error occurred
	 */
	@SafeVarargs
	public final void createClassFromTemplate(File javaSrcFolder, String templatePath, String packageName,
			String className, boolean overwrite, TemplateVariable... vars) throws IOException {
		File packageFolder = createPackageFolder(javaSrcFolder, packageName);

		File classFile = new File(packageFolder, className + ".java");
		if (!classFile.exists()) {
			Files.createFile(classFile.toPath());
		} else if (!overwrite) {
			throw new IOException(format("Class file %s already exists.", classFile));
		}

		/* lets make a context and put data into it */
		Map<String, String> context = new HashMap<>();
		context.put("package", packageName);
		context.put("name", className);

		for (TemplateVariable variable : vars) {
			context.put(variable.getKey(), variable.getValue());
		}

		createFileFromTemplate(templatePath, classFile, context);
	}

	/**
	 * Creates a new file from a specified resource template.
	 *
	 * @param templatePath resource path of the template to use
	 * @param targetFolder target folder of the new file
	 * @param overwrite    use <code>true</code> if existing file shall be overwritten
	 * @param variables    additional array of key/value pairs used to replace variables in template
	 * @throws IOException if an I/O error occurred
	 */
	@SafeVarargs
	public final void createFileFromTemplate(String templatePath, File targetFolder, boolean overwrite,
			TemplateVariable... variables) throws IOException {
		File outputFile = new File(targetFolder, templatePath);

		if (!outputFile.exists()) {
			Utils.prepareDestination(outputFile, false);
			Files.createFile(outputFile.toPath());
		} else if (!overwrite) {
			throw new IOException(format("File %s already exists.", outputFile));
		}

		/* lets make a context and put data into it */
		Map<String, String> context = new HashMap<>();
		for (TemplateVariable variable : variables) {
			context.put(variable.getKey(), variable.getValue());
		}

		createFileFromTemplate(templatePath, outputFile, context);
	}

	private void createFileFromTemplate(String templatePath, File outputFile, Map<String, String> context)
			throws IOException {
		try (InputStream input = this.getClass().getResourceAsStream(templatePath)) {
			// get requested template
			String template = IOUtils.toString(input, StandardCharsets.UTF_8.name());

			// fill template with given parameters
			for (Entry<String, String> variable : context.entrySet()) {
				String varName = "${" + variable.getKey() + "}";
				template = template.replace(varName, variable.getValue());
			}

			// write filled template into file
			try (ByteArrayInputStream byteIn = new ByteArrayInputStream(template.getBytes(StandardCharsets.UTF_8));
					FileOutputStream fileOut = new FileOutputStream(outputFile)) {
				IOUtils.copy(byteIn, fileOut);
			}
		}
	}

	private static File createPackageFolder(File sourceFolder, String packageName) throws IOException {
		File packageFolder = Utils.getPackageDir(sourceFolder, packageName);
		Utils.prepareDestination(packageFolder, true);
		return packageFolder;
	}

	/**
	 * Holds key and value of a template variable. The corresponding variable in template will be replaced with the
	 * value of this instance.
	 */
	public static class TemplateVariable extends AbstractMap.SimpleEntry<String, String> {
		private static final long serialVersionUID = 4442052075055129731L;

		private TemplateVariable(String key, String value) {
			super(key, value);
		}

		public static TemplateVariable of(String key, String value) {
			Objects.nonNull(key); // at least key may not null
			return new TemplateVariable(key, value);
		}
	}
}
