/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.build;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.exec.CommandLine;
import org.apache.commons.exec.ExecuteException;
import org.apache.commons.io.output.TeeOutputStream;
import org.apache.maven.plugins.annotations.Parameter;
import org.codehaus.plexus.util.StringUtils;
import org.eclipse.aether.RepositorySystemSession;

import com.sap.cds.maven.plugin.AbstractCdsMojo;
import com.sap.cds.maven.plugin.util.ProcessExecutor;

/**
 * An abstract base class for all Node.js related CDS Mojos.
 */
abstract class AbstractNodejsMojo extends AbstractCdsMojo {
	protected static final String PROP_NODE_DIR = "cds.node.directory";
	protected static final String PROP_NODE_EXECUTABLE = "cds.node.executable";
	protected static final String PROP_NPM_EXECUTABLE = "cds.npm.executable";
	protected static final String PROP_NPX_EXECUTABLE = "cds.npx.executable";

	protected static final String PARAM_NODE_DIR = "${" + PROP_NODE_DIR + "}";
	protected static final String PARAM_NPX_EXECUTABLE = "${" + PROP_NPX_EXECUTABLE + "}";
	protected static final String PARAM_NPM_EXECUTABLE = "${" + PROP_NPM_EXECUTABLE + "}";

	/**
	 * Additional environment variables to set on the command line.
	 *
	 * @since 1.26.0
	 */
	@Parameter
	private Map<String, String> environmentVariables;

	/**
	 * URL of NPM registry to use.
	 */
	@Parameter(property = "cds.npm.registry")
	private String npmRegistry;

	/**
	 * Defines settings and components that control the repository system.
	 */
	@Parameter(defaultValue = "${repositorySystemSession}", required = true, readonly = true)
	protected RepositorySystemSession repositorySystemSession;

	// internally used parameter, not configurable in pom.xml
	@Parameter(defaultValue = PARAM_NODE_DIR, readonly = true)
	private File nodeDir;

	/**
	 * Executes a program as child process.
	 *
	 * @param workDir       the working directory of child process
	 * @param execFile      the executable to use
	 * @param outputStream  an optional standard output stream
	 * @param additionalEnv additional environment variables, can be <code>null</code>
	 * @param exitValues    an optional list with exit values of the process to be considered successful, can be
	 *                      <code>null</code>
	 * @param args          an optional command line arguments passed to the process
	 * @throws IOException      if an I/O exception occurred
	 * @throws ExecuteException if process execution failed
	 */
	protected void execute(File workDir, File execFile, OutputStream outputStream, Map<String, String> additionalEnv, int[] exitValues,
			String... args) throws IOException {

		// just use executable name to support cli mvn com.sap.cds:cds-maven-plugin:1.6.0-SNAPSHOT:cds
		// -Darguments=build/all were node-install was not called previously
		CommandLine cmdLine = execFile.isAbsolute() ? new CommandLine(execFile) : new CommandLine(execFile.getName());
		cmdLine.addArguments(args, false);

		List<String> paths = this.nodeDir != null ? Collections.singletonList(this.nodeDir.getAbsolutePath()) : Collections.emptyList();
		ProcessExecutor executor = new ProcessExecutor(workDir, cmdLine, getEnvironment(additionalEnv), paths);

		// set optional list with exit value of the process to be considered successful
		executor.setExitValues(exitValues);
		logInfo("Executing %s in working directory %s", strong(cmdLine), strong(workDir));

		if (outputStream != null) {
			ByteArrayOutputStream standardByteOut = new ByteArrayOutputStream();

			try (OutputStream standardOut = new TeeOutputStream(outputStream, standardByteOut)) {
				executor.execute(standardOut, standardOut, this);
			} catch (ExecuteException ex) {
				// write standard and error output as error into Maven log
				logError(new String(standardByteOut.toByteArray(), StandardCharsets.UTF_8));
				throw ex;
			}
		} else {
			logInfo(executor.executeAndGetResult(this));
		}
	}

	/**
	 * Finds the cds working directory by going upwards and looking for a .cdsrc.json file. It stops at the top-most project
	 * directory.
	 *
	 * @return the cds working directory or <code>null</code> if not found.
	 */
	protected File findCdsWorkingDir() {
		File currentDir = super.project.getBasedir();
		File reactorBaseDir = getReactorBaseDirectory();
		String reactorAbsolutePath = reactorBaseDir.getAbsolutePath();

		while (currentDir != null && currentDir.getAbsolutePath().startsWith(reactorAbsolutePath)) {

			// check if current directory contains a .cdsrc.json or package.json
			File cdsrcJsonFile = new File(currentDir, ".cdsrc.json");
			if (cdsrcJsonFile.exists()) {
				logInfo("Using directory containing a .cdsrc.json as working directory: %s", strong(currentDir));
				return cdsrcJsonFile.getParentFile();
			}

			File packageJsonFile = new File(currentDir, "package.json");
			if (packageJsonFile.exists()) {
				logInfo("Using directory containing a package.json as working directory: %s", strong(currentDir));
				return packageJsonFile.getParentFile();
			}

			// nothing found yet, go up one directory level
			currentDir = currentDir.getParentFile();
		}

		logInfo("Using topmost project directory as working directory: %s", strong(currentDir));
		return reactorBaseDir;
	}

	private Map<String, String> getEnvironment(Map<String, String> additionalEnv) {
		Map<String, String> environment = new HashMap<>();
		if (this.environmentVariables != null) {
			environment.putAll(this.environmentVariables);
		}
		// use configured NPM registry if available
		if (StringUtils.isNotBlank(this.npmRegistry)) {
			environment.put("NPM_CONFIG_REGISTRY", this.npmRegistry);
			logInfo("Using npm repository: %s", this.npmRegistry);
		}
		if (additionalEnv != null) {
			environment.putAll(additionalEnv);
		}
		return environment;
	}

}
