/**************************************************************************
 * (C) 2019-2023 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import static java.util.Objects.nonNull;

import java.util.HashMap;
import java.util.Map;
import java.util.function.BiPredicate;

import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.ArtifactUtils;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.DefaultProjectBuildingRequest;
import org.apache.maven.project.MavenProject;
import org.apache.maven.project.ProjectBuildingRequest;
import org.apache.maven.shared.dependency.graph.DependencyGraphBuilder;
import org.apache.maven.shared.dependency.graph.DependencyGraphBuilderException;
import org.apache.maven.shared.dependency.graph.DependencyNode;
import org.apache.maven.shared.dependency.graph.traversal.DependencyNodeVisitor;
import org.eclipse.aether.RepositorySystemSession;

public class DependencyFinder implements DependencyNodeVisitor {

	private final Map<String, Artifact> artifacts;
	private final BiPredicate<String, String> artifactFilter;

	public static DependencyFinder scanDependencyGraph(
			MavenProject project,
			DependencyGraphBuilder dependencyGraphBuilder,
			RepositorySystemSession repositorySystemSession,
			BiPredicate<String, String> artifactFilter) throws MojoExecutionException {

		DependencyFinder finder = new DependencyFinder(artifactFilter);
		ProjectBuildingRequest request = new DefaultProjectBuildingRequest();
		request.setProject(project);
		request.setRepositorySession(repositorySystemSession);

		try {
			DependencyNode graph = dependencyGraphBuilder.buildDependencyGraph(request, null);
			graph.accept(finder);
			return finder;
		} catch (DependencyGraphBuilderException e) {
			throw new MojoExecutionException(e.getMessage(), e);
		}
	}

	private DependencyFinder(BiPredicate<String, String> artifactFilter) {
		this.artifacts = new HashMap<>();
		this.artifactFilter = artifactFilter;
	}

	private boolean shouldAddArtifact(String groupId, String artifactId) {
		if (nonNull(this.artifactFilter)) {
			return this.artifactFilter.test(groupId, artifactId);
		}

		return true;
	}

	@Override
	public boolean visit(DependencyNode node) {
		Artifact artifact = node.getArtifact();

		if (shouldAddArtifact(artifact.getGroupId(), artifact.getArtifactId())) {
			String artifactKey = ArtifactUtils.versionlessKey(artifact);
			this.artifacts.put(artifactKey, artifact);
		}

		/*
			* Return <code>true</code> to visit the specified dependency node's children, <code>false</code> to skip
			* the specified dependency node's children and proceed to its next sibling
			*/
		return true;
	}

	@Override
	public boolean endVisit(DependencyNode node) {
		/*
			* Return <code>true</code> to visit the specified dependency node's next sibling, <code>false</code> to
			* skip the specified dependency node's next siblings and proceed to its parent
			*/
		return true;
	}

	public Artifact getArtifact(String groupId, String artifactId) {
		return this.artifacts.get(ArtifactUtils.versionlessKey(groupId, artifactId));
	}
}
