/**************************************************************************
 * (C) 2019-2022 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.maven.plugin.util;

import static java.io.File.separator;

import java.util.Locale;

/**
 * This enum contains all supported platforms.
 */
public enum Platform {
	/**
	 * Running on a Linux operating system
	 */
	LINUX("linux", "tar.gz", "node", "npm", "npx", "bin"),

	/**
	 * Running on Mac OS (Darwin)
	 */
	MAC("darwin", "tar.gz", "node", "npm", "npx", "bin"),

	/**
	 * Running on a Windows operating system
	 */
	WINDOWS("win", "zip", "node.exe", "npm.cmd", "npx.cmd", null);

	/**
	 * Contains the current {@link Platform}.
	 */
	public static final Platform CURRENT = guess();

	/**
	 * Contains the Node.js archive extension.
	 */
	public final String archiveExtension;

	/**
	 * Contains name of node executable, either "node" or "node.exe".
	 */
	public final String node;

	/**
	 * Contains the Node.js classifier: "linux-x64", "darwin-x64", "linux-arm64", "darwin-arm64" or "win-x64"
	 */
	public final String nodeClassifier;

	/**
	 * Contains name of npm executable, either "npm" or "npm.cmd".
	 */
	public final String npm;

	/**
	 * Contains name of npx executable, either "npx" or "npx.cmd".
	 */
	public final String npx;

	/**
	 * Contains relative path of node executable within the extracted Node.js archive.
	 */
	private final String nodePath;

	/**
	 * Contains relative path of npm executable within the extracted Node.js archive.
	 */
	private final String npmPath;

	/**
	 * Contains relative path of npx executable within the extracted Node.js archive.
	 */
	private final String npxPath;

	private Platform(String codename, String archiveExtension, String node, String npm, String npx, String path) {
		String osArch = System.getProperty("os.arch").toLowerCase();
		String suffix = osArch.contains("aarch64") ? "-arm64" : "-x64";
		this.nodeClassifier = codename + suffix;
		this.archiveExtension = archiveExtension;
		this.node = node;
		this.npm = npm;
		this.npx = npx;

		if (path != null) {
			this.nodePath = path + separator + node;
			this.npmPath = path + separator + npm;
			this.npxPath = path + separator + npx;
		} else {
			this.nodePath = node;
			this.npmPath = npm;
			this.npxPath = npx;
		}
	}

	/**
	 * Calculates the relative Node.js download path.
	 * 
	 * @param nodeVersion the Node.js version
	 * @return the calculated relative download path
	 */
	public String getDownloadPath(String nodeVersion) {
		return nodeVersion + "/" + getLongDirectoryName(nodeVersion) + "." + this.archiveExtension;
	}

	/**
	 * @param nodeVersion the Node.js version
	 * @return a relative path to node executable for the current platform
	 */
	public String getNodePath(String nodeVersion) {
		return getLongDirectoryName(nodeVersion) + separator + this.nodePath;
	}

	/**
	 * @param nodeVersion the Node.js version
	 * @return a relative path to npm executable for the current platform
	 */
	public String getNpmPath(String nodeVersion) {
		return getLongDirectoryName(nodeVersion) + separator + this.npmPath;
	}

	/**
	 * @param nodeVersion the Node.js version
	 * @return a relative path to npx executable for the current platform
	 */
	public String getNpxPath(String nodeVersion) {
		return getLongDirectoryName(nodeVersion) + separator + this.npxPath;
	}

	/**
	 * @return <code>true</code> if the current platform is {@link Platform#WINDOWS}.
	 */
	public boolean isWindows() {
		return this == WINDOWS;
	}

	/**
	 * @param nodeVersion the Node.js version
	 * @return the long Node.js directory name
	 */
	private String getLongDirectoryName(String nodeVersion) {
		return "node-" + nodeVersion + "-" + this.nodeClassifier;
	}

	private static Platform guess() {
		String osName = System.getProperty("os.name");
		if (osName == null) {
			throw new IllegalStateException("Couldn't get OS name.");
		}
		osName = osName.toUpperCase(Locale.ENGLISH);
		if (osName.contains("WINDOWS"))
			return WINDOWS;
		return osName.contains("MAC") ? MAC : LINUX;
	}
}
