/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.changeset;

import com.sap.cds.services.persistence.PersistenceService;
import com.sap.cds.services.transaction.ChangeSetMemberDelegate;
import com.sap.cds.services.transaction.TransactionManager;

/**
 * A {@link ChangeSetMember} links a certain resource to a {@link ChangeSetContext}.
 * It defines how this resource is completed (committed) or cancelled (rolled back).
 * A common implementation of a {@link ChangeSetMember} is the {@link ChangeSetMemberDelegate}, which wraps a given {@link TransactionManager}.
 */
public interface ChangeSetMember {

	/**
	 * The name should be directly linked to the resource the {@link ChangeSetMember} manages, e.g. the {@link PersistenceService} name.
	 * It can be used to ensure that a resource is only registered once on a given {@link ChangeSetContext}.
	 * @return the name of the {@link ChangeSetMember}
	 */
	String getName();

	/**
	 * Defines how the underlying resource of this {@link ChangeSetMember} can be completed (comitted)
	 */
	void complete();

	/**
	 * Defines how the underlying resource of this {@link ChangeSetMember} can be cancelled (rolled back)
	 */
	void cancel();

	/**
	 * Defines, if the underlying resource of this {@link ChangeSetMember} was requested to be cancelled (rolled back).
	 * Please note that usually a {@link ChangeSetContext} is marked for cancel as a whole through {@link ChangeSetContext#markForCancel()}
	 * This method is mainly intended if there is some other (outside) API available, that could set the cancellation marker on the underlying resource.
	 * @return true, if the underlying resource was marked for cancellation through some (outside) API.
	 */
	boolean isMarkedForCancel();

}
