/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.mt;

import java.util.Arrays;
import java.util.List;
import java.util.Map;

import com.sap.cds.services.Service;

/**
 * Implements steps during subscriptions, upgrades and unsubscription of tenants
 */
public interface DeploymentService extends Service {

	String DEFAULT_NAME = "DeploymentService$Default";
	String EVENT_DEPENDENCIES = "DEPENDENCIES";
	String EVENT_SUBSCRIBE = "SUBSCRIBE";
	String EVENT_UPGRADE = "UPGRADE";
	String EVENT_UNSUBSCRIBE = "UNSUBSCRIBE";

	/**
	 * Returns a list of dependencies required by this application.
	 * When reporting dependencies to SaaS Registry consider wrapping the {@link Map} with {@code SaasRegistryDependency}, as a type-safe alternative.
	 *
	 * @return the list of required dependencies
	 */
	List<Map<String, Object>> dependencies();

	/**
	 * Performs the subscription
	 * @param tenant the tenant ID
	 * @param options optional additional options
	 */
	void subscribe(String tenant, Map<String, Object> options);

	/**
	 * Performs the upgrade
	 * @param tenant the tenant ID
	 * @param options optional additional options
	 */
	default void upgrade(String tenant, Map<String, Object> options) {
		upgrade(Arrays.asList(tenant), options);
	}

	/**
	 * Performs the upgrade
	 * @param tenants the list of tenant IDs
	 * @param options optional additional options
	 */
	void upgrade(List<String> tenants, Map<String, Object> options);

	/**
	 * Performs the unsubscription
	 * @param tenant the tenant ID
	 * @param options optional additional options
	 */
	void unsubscribe(String tenant, Map<String, Object> options);

}
