/**************************************************************************
 * (C) 2019-2024 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.outbox;

import java.util.Map;

import com.sap.cds.CdsData;
import com.sap.cds.Struct;

/**
 * Represents the default structure of a message stored in the outbox and handled generically.
 * Additional information can be freely added to the underlying {@link Map} structure.
 */
public interface OutboxMessage extends CdsData {

	String EVENT = "event";
	String PARAMS = "params";

	/**
	 * Creates a new {@link OutboxMessage}
	 * @return the {@link OutboxMessage}
	 */
	static OutboxMessage create() {
		return Struct.create(OutboxMessage.class);
	}

	/**
	 * @return the original event stored in the outbox
	 */
	String getEvent();

	/**
	 * Sets the original event, to be stored in the outbox
	 * @param event the event
	 */
	void setEvent(String event);

	/**
	 * @return the name of the entity that the event is related to
	 */
	String getEntity();

	/**
	 * Sets the name of the entity that the event is related to
	 * @param entity the entity name
	 */
	void setEntity(String entity);

	/**
	 * @return the event context parameters of the original event
	 */
	Map<String, Object> getParams();

	/**
	 * Sets the event context parameters of the original event
	 * @param data the context parameters
	 */
	void setParams(Map<String, Object> data);

	/**
	 * @return the request context information stored in the outbox
	 */
	StoredRequestContext getStoredRequestContext();

	/**
	 * Sets the request context information, to be stored in the outbox.
	 * The values stored in this map are used to restore the request context when the outbox message is published.
	 *
	 * @param storedRequestContext the request context information to be stored in the outbox
	 */
	void setStoredRequestContext(StoredRequestContext storedRequestContext);

}
