package com.sap.cds.services;

import java.util.List;
import java.util.Map;

import com.sap.cds.services.changeset.ChangeSetContext;
import com.sap.cds.services.changeset.ChangeSetContextAccessor;
import com.sap.cds.services.changeset.ChangeSetContextSPI;
import com.sap.cds.services.environment.CdsProperties;
import com.sap.cds.services.environment.PropertiesProvider;
import com.sap.cds.services.messages.Message;
import com.sap.cds.services.messages.Message.Severity;
import com.sap.cds.services.request.FeatureToggle;
import com.sap.cds.services.request.FeatureTogglesInfo;
import com.sap.cds.services.request.ModifiableParameterInfo;
import com.sap.cds.services.request.ModifiableUserInfo;
import com.sap.cds.services.request.ParameterInfo;
import com.sap.cds.services.request.RequestContext;
import com.sap.cds.services.request.RequestContextAccessor;
import com.sap.cds.services.request.UserInfo;
import com.sap.cds.services.runtime.CdsRuntimeConfigurer;
import com.sap.cds.services.runtime.ExtendedServiceLoader;

/**
 * Factory to create various objects for API interfaces.
 * This class should never be used directly by application code and is only intended for internal usage in cds-services-api.
 * Applications rather should use the specific {@code create} methods on the target interfaces.
 *
 * The implementation of the factory is loaded via {@link ExtendedServiceLoader} and is therefore pluggable.
 */
public interface CoreFactory {

	/**
	 * The instance loaded via {@link ExtendedServiceLoader}.
	 */
	CoreFactory INSTANCE = ExtendedServiceLoader.loadSingle(CoreFactory.class);

	/**
	 * The factory method to create a new {@link EventContext}.
	 *
	 * Application code should use {@link EventContext#create(String, String)} instead.
	 *
	 * @param event the name of the event
	 * @param entityName the name of the entity
	 * @return the {@link EventContext}
	 */
	EventContext createEventContext(String event, String entityName);

	/**
	 * The factory method to create a new {@link Service}.
	 *
	 * Application code should use {@link EventContext#create(String, String)} instead.
	 *
	 * @param name the name of the service
	 * @return the {@link Service}
	 */
	Service createService(String name);

	/**
	 * The factory method to create a new {@link Service} with a given delegator {@link Service}.
	 *
	 * This method is intended to be used only by {@link ServiceDelegator}.
	 *
	 * @param name the name of the service
	 * @param delegator the delegator {@link Service}
	 * @return the {@link Service}
	 */
	Service createService(String name, Service delegator);

	/**
	 * The factory method to create a new {@link CdsRuntimeConfigurer} with the given properties.
	 *
	 * Application code should use {@link CdsRuntimeConfigurer#create()} or {@link CdsRuntimeConfigurer#create(PropertiesProvider)} instead.
	 *
	 * @param propertiesProvider the {@link PropertiesProvider},
	 * 		may be {@code null}, which means a simple {@link PropertiesProvider} providing default {@link CdsProperties} is used
	 *
	 * @return the {@link CdsRuntimeConfigurer}
	 */
	CdsRuntimeConfigurer createCdsRuntimeConfigurer(PropertiesProvider propertiesProvider);

	/**
	 * The factory method to create a new {@link RequestContextAccessor}.
	 *
	 * This method is intended to be used only by {@link RequestContext}.
	 *
	 * @return the {@link RequestContextAccessor}
	 */
	RequestContextAccessor createRequestContextAccessor();

	/**
	 * The factory method to create a new {@link ChangeSetContextAccessor}.
	 *
	 * This method is intended to be used only by {@link ChangeSetContext}.
	 *
	 * @return the {@link ChangeSetContextAccessor}
	 */
	ChangeSetContextAccessor createChangeSetContextAccessor();

	/**
	 * The factory method to create a new {@link ChangeSetContextSPI}.
	 *
	 * Application code should use {@link ChangeSetContextSPI#open()} instead.
	 *
	 * @return the {@link ChangeSetContextSPI}
	 */
	ChangeSetContextSPI createChangeSetContextSPI();

	/**
	 * Creates a {@link ModifiableParameterInfo} based on the passed {@link ParameterInfo}.
	 * If {@code parameterInfo} is {@code null}, a {@link ModifiableParameterInfo} based on
	 * default values of a clear {@link ParameterInfo} is returned.
	 *
	 * Application code should use {@link ParameterInfo#create()} or {@link ParameterInfo#copy()} instead.
	 *
	 * @param parameterInfo	The {@link ParameterInfo} as basis.
	 * @return	The created {@link ModifiableParameterInfo} instance.
	 */
	ModifiableParameterInfo createParameterInfo(ParameterInfo parameterInfo);

	/**
	 * Creates a {@link ModifiableUserInfo} based on the passed {@link UserInfo}.
	 * If {@code userInfo} is {@code null}, a {@link ModifiableUserInfo} based on
	 * default values of a clear {@link UserInfo} is returned.
	 *
	 * Application code should use {@link UserInfo#create()} or {@link UserInfo#copy()} instead.
	 *
	 * @param userInfo	The {@link UserInfo} as basis.
	 * @return	The created {@link ModifiableUserInfo} instance.
	 */
	ModifiableUserInfo createUserInfo(UserInfo userInfo);

	/**
	 * Creates a {@link FeatureTogglesInfo} object.
	 *
	 * Application code should use {@link FeatureTogglesInfo#create(List)} instead.
	 *
	 * @param featureToggles A {@link List} of feature toggles
	 * @return A {@link FeatureTogglesInfo} object containing all provided feature toggles in the stream
	 */
	FeatureTogglesInfo createFeatureTogglesInfo(List<FeatureToggle> featureToggles);

	/**
	 * Creates a {@link FeatureTogglesInfo} object.
	 *
	 * Application code should use {@link FeatureTogglesInfo#create(Map)} instead.
	 *
	 * @param featureToggles A {@link Map} where the key is the feature name and the value indicates if the feature is enable or not
	 * @return A {@link FeatureTogglesInfo} object containing the feature toggles created from the map entries
	 */
	FeatureTogglesInfo createFeatureTogglesInfo(Map<String, Boolean> featureToggles);

	/**
	 * The factory method to create a new {@link ServiceExceptionUtils}.
	 *
	 * This method is intended to be used only by {@link ServiceException}.
	 *
	 * @return the {@link ServiceExceptionUtils}
	 */
	ServiceExceptionUtils createServiceExceptionUtils();

	/**
	 * Creates a {@link Message} object.
	 *
	 * Application code should use {@link Message#create(Severity, String)} instead.
	 *
	 * @param severity the {@link Severity} of the {@link Message}
	 * @param text the final text of the message
	 * @return the {@link Message}
	 */
	Message createMessage(Severity severity, String text);

	/**
	 * Creates a {@link Message} object.
	 *
	 * Application code should use {@link Message#create(String, ServiceException)} instead.
	 *
	 * @param text the final text of the message
	 * @param exception the {@link ServiceException} to copy optional arguments from
	 * @return the {@link Message}
	 */
	Message createMessage(String text, ServiceException exception);

	/**
	 * Creates a {@link Message} object.
	 *
	 * Application code should use {@link Message#create(Severity, String, Message)} instead.
	 *
	 * @param severity the {@link Severity} of the {@link Message}
	 * @param text the final text of the message
	 * @param message the {@link Message}
	 * @return the {@link Message}
	 */
	Message createMessage(Severity severity, String text, Message message);

}
