/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services;

/**
 * This enum implements {@link ErrorStatus}, which is the interface required by
 * {@link ServiceException}. It provides a default set of {@link ErrorStatus}
 * constants.
 *
 * Additional {@link ErrorStatus} can be defined, by either implementing
 * classes, which implement {@link ErrorStatus} or by defining an additional
 * enum, which again implements the interface and defines further constants.
 */
public enum ErrorStatuses implements ErrorStatus {

	// DEFAULT HTTP ERROR CODES
	BAD_REQUEST(400, "Bad request", 400),

	UNAUTHORIZED(401, "No authentication", 401), // == unauthenticated (misleading http name)

	FORBIDDEN(403, "No authorization", 403), // == unauthorized

	NOT_FOUND(404, "Not found", 404),

	METHOD_NOT_ALLOWED(405, "Method not allowed", 405),

	NOT_ACCEPTABLE(406, "Not acceptable", 406),

	CONFLICT(409, "Conflict", 409),

	GONE(410, "Gone", 410),

	PRECONDITION_FAILED(412, "Precondition failed", 412),

	UNSUPPORTED_MEDIA_TYPE(415, "Unsupported media type", 415),

	MISDIRECTED_REQUEST(421, "Misdirected request", 421),

	UNPROCESSABLE_ENTITY(422, "Unprocessable entity", 422),

	LOCKED(423, "Locked", 423),

	PRECONDITION_REQUIRED(428, "Precondition required", 428),

	SERVER_ERROR(500, "Internal server error", 500),

	NOT_IMPLEMENTED(501, "Not implemented", 501),

	BAD_GATEWAY(502, "Bad gateway", 502),

	GATEWAY_TIMEOUT(504, "Gateway timeout", 504);

	private final int code;
	private final String description;
	private final int httpStatus;

	private ErrorStatuses(int code, String description, int httpStatus) {
		this.code = code;
		this.description = description;
		this.httpStatus = httpStatus;
	}

	@Override
	public String getDescription() {
		return description;
	}

	@Override
	public int getHttpStatus() {
		return httpStatus;
	}

	/**
	 * @param code the code
	 * @return the ErrorStatus from this enum, associated with the given code or {@code null}
	 */
	public static ErrorStatus getByCode(int code) {
		for(ErrorStatus errorStatus : ErrorStatuses.values()) {
			if(errorStatus.getHttpStatus() == code) {
				return errorStatus;
			}
		}
		return null;
	}

	@Override
	public String getCodeString() {
		return Integer.toString(code);
	}

}
