/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services;

import java.util.Locale;
import java.util.function.Function;

import com.sap.cds.ql.StructuredType;
import com.sap.cds.ql.cqn.Path;
import com.sap.cds.reflect.CdsElement;
import com.sap.cds.services.environment.CdsProperties.Errors;
import com.sap.cds.services.messages.MessageTarget;
import com.sap.cds.services.request.RequestContext;

/**
 * Helper methods for creating (localized) messages in {@link ServiceException}.
 * <p>
 * {@code ServiceExceptionUtils#getMessage} is used to implement {@code ServiceException#getMessage} and
 * {@code ServiceExceptionUtils#getLocalizedMessage} is used to implement {@code ServiceException#getLocalizedMessage} respectively.
 */
public interface ServiceExceptionUtils {

	/**
	 * Creates a message based on code and args. {@code code} may either be a SL4J-message format or a bundle key.
	 * The JVM's default locale is taken to lookup code in resource bundles.
	 *
	 * @param code	The used code
	 * @param args	The arguments for message formats.
	 * @return	The created message
	 */
	String getMessage(String code, Object[] args);

	/**
	 * Creates a message based on code and args. {@code code} may either be a SL4J-message format or a bundle key.
	 * The specified locale is taken to lookup the code in resource bundles.
	 * If the specified locale is <code>null</code> the locale provided of the current {@link RequestContext} is used
	 *
	 * @param code	 The used code
	 * @param args	 The arguments for message formats.
	 * @param locale The locale
	 * @return	The created message
	 */
	String getLocalizedMessage(String code, Object[] args, Locale locale);

	/**
	 * Creates a new {@link MessageTarget} with a string-based target
	 *
	 * @param target the string-based target
	 * @return the {@link MessageTarget}
	 */
	MessageTarget getMessageTarget(String target);

	/**
	 * Creates a new {@link MessageTarget} with a target, based on a target
	 * parameter and a CQN path.
	 *
	 * @param parameter target parameter serving as the entry point for the
	 *                        path resolution
	 * @param path            the path to the target
	 * @return the {@link MessageTarget}
	 */
	MessageTarget getMessageTarget(String parameter, Function<StructuredType<?>, Object> path);

	/**
	 * Creates a new {@link MessageTarget} with a target, based on a target
	 * parameter and a CQN path.
	 *
	 * @param parameter target parameter serving as the entry point for the
	 *                        path resolution
	 * @param type            the root type of the path. Either an entity or a
	 *                        structured type.
	 * @param path            the path to the target
	 * @param <E>             the entity type
	 * @return the {@link MessageTarget}
	 */
	<E extends StructuredType<E>> MessageTarget getMessageTarget(String parameter, Class<E> type, Function<E, Object> path);

	/**
	 * Creates a new {@link MessageTarget} with a target, based on a {@link Path}
	 * and {@link CdsElement}.
	 *
	 * @param path the cqn path to the targeted cds element
	 * @param element the targeted element
	 * @return the {@link MessageTarget}
	 */
	MessageTarget getMessageTarget(Path path, CdsElement element);

	/**
	 * @return the active {@link Errors} properties
	 */
	Errors getErrorsProperties();
}
