/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.authentication;

/**
 * The {@link AuthenticationInfo} provides access to the authorization of the current request.
 * Usually {@link com.sap.cds.services.runtime.UserInfoProvider} react on a specific {@link AuthenticationInfo} implementations to extract user (principal) information from the authentication.
 * The {@link AuthenticationInfo} can also be used to enable principal propagation of the current user in the request to a different service, as it provides access to the original authentication.
 */
public interface AuthenticationInfo {

	/**
	 * Can be used to check if an {@link AuthenticationInfo} is an instance of the common {@link AuthenticationInfo} implementations:
	 * - {@link BasicAuthenticationInfo}
	 * - {@link JwtTokenAuthenticationInfo}
	 * - {@link JwtTokenWithForwardJwtAuthenticationInfo}
	 *
	 * @param clazz the class to check the instance against
	 * @return true, if the {@link AuthenticationInfo} is of an instance of {@code clazz}
	 */
	default boolean is(Class<? extends AuthenticationInfo> clazz) {
		return clazz.isAssignableFrom(this.getClass());
	}

	/**
	 * Can be used to cast the {@link AuthenticationInfo} to a specific implementation:
	 * - {@link BasicAuthenticationInfo}
	 * - {@link JwtTokenAuthenticationInfo}
	 * - {@link JwtTokenWithForwardJwtAuthenticationInfo}
	 *
	 * @param <T> the type of the class
	 * @param clazz the class to cast the instance to
	 * @return the instance casted to type {@code T}
	 */
	default <T extends AuthenticationInfo> T as(Class<T> clazz) {
		return clazz.cast(this);
	}

}
