/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.authorization;

import com.sap.cds.ql.cqn.CqnPredicate;
import com.sap.cds.ql.cqn.CqnStatement;
import com.sap.cds.services.Service;

/**
 * Technical service for checking authorization on service or entity level for the current user.
 */
public interface AuthorizationService extends Service {

	static final String DEFAULT_NAME = "AuthorizationService$Default";

	static final String EVENT_SERVICE_ACCESS = "SERVICE_ACCESS";

	static final String EVENT_ENTITY_ACCESS = "ENTITY_ACCESS";

	static final String EVENT_FUNCTION_ACCESS = "FUNCTION_ACCESS";

	static final String EVENT_ACTION_ACCESS = "ACTION_ACCESS";

	static final String EVENT_CALC_WHERE_CONDITION = "CALC_WHERE_CONDITION";

	/**
	 * Checks if the current user is allowed to access the given service.
	 * Services might be restricted on basis of roles and events.
	 *
	 * @param serviceName	The (fully-qualified) name of the service
	 * @param event		The event
	 * @return	{@code true} in case of permission, {@code false} otherwise
	 */
	boolean hasServiceAccess(String serviceName, String event);

	/**
	 * Checks if the current user is allowed to access the given entity.
	 * Entities might be restricted on basis of roles, events and instance based conditions.
	 *
	 * @param entityName	The (fully-qualified) name of the entity
	 * @param event		The event
	 * @return	{@code true} in case of permission, {@code false} otherwise
	 */
	boolean hasEntityAccess(String entityName, String event);

	/**
	 * Checks if the current user is allowed to access the entity via the given query.
	 * Authorizations for auto-exposed entities may be derived from an entity on the path.
	 * Entities might be restricted on basis of roles, events and instance based conditions.
	 *
	 * @param entityName	The (fully-qualified) name of the entity
	 * @param event		The event
	 * @param query		The query defining the access path
	 * @return	{@code true} in case of permission, {@code false} otherwise
	 */
	boolean hasEntityAccess(String entityName, String event, CqnStatement query);

	/**
	 * Checks if the current user is allowed to access the given function.
	 * Functions might be restricted on basis of roles or instance based conditions.
	 *
	 * @param entityName	The (fully-qualified) name of the entity in case of bound function
	 * @param functionName 	The name of the function, full-qualified in case of unbound function

	 * @return	{@code true} in case of permission, {@code false} otherwise
	 */
	boolean hasFunctionAccess(String entityName, String functionName);

	/**
	 * Checks if the current user is allowed to access the given action.
	 * Actions might be restricted on basis of roles or instance based conditions.
	 *
	 * @param entityName	The (fully-qualified) name of the entity in case of bound action
	 * @param actionName 	The name of the action, full-qualified in case of unbound action
	 *
	 * @return	{@code true} in case of permission, {@code false} otherwise
	 */
	boolean hasActionAccess(String entityName, String actionName);

	/**
	 * Calculates a {@link CqnPredicate} which reflects the 'where' conditions for the given entity and event.
	 *
	 * @param entityName	The (fully-qualified) name of the entity
	 * @param event	The event
	 * @return	The predicate or {@code null} if there is no limitation
	 */
	CqnPredicate calcWhereCondition(String entityName, String event);
}
