/**************************************************************************
 * (C) 2019-2021 SAP SE or an SAP affiliate company. All rights reserved. *
 **************************************************************************/
package com.sap.cds.services.changeset;

import com.sap.cds.services.CoreFactory;
import com.sap.cds.services.Service;
import com.sap.cds.services.handler.Handler;

/**
 * A {@link ChangeSetContext} defines the transactional boundaries around one or multiple {@link Service#emit(com.sap.cds.services.EventContext)} calls.
 * It can be used by {@link Handler} or other participants to ensure that changes, which were done as part of a ChangeSetContext are canceled (rolled back).
 * <br>
 * In addition it provides the ability to perform actions at the end of a ChangeSet by registering a {@link ChangeSetListener}
 */
public interface ChangeSetContext {

	/**
	 * @return true, if a {@link ChangeSetContext} is currently active
	 */
	static boolean isActive() {
		return CoreFactory.INSTANCE.createChangeSetContextAccessor().isActive();
	}

	/**
	 * Gives access to the currently active {@link ChangeSetContext}.
	 * If no {@link ChangeSetContext} is currently opened this method will return <code>null</code>
	 *
	 * @return the currently active {@link ChangeSetContext} or <code>null</code>.
	 */
	static ChangeSetContext getCurrent() {
		return CoreFactory.INSTANCE.createChangeSetContextAccessor().getCurrent();
	}

	/**
	 * @return an numeric identifier of the {@link ChangeSetContext}, used for logging
	 */
	int getId();

	/**
	 * Marks the {@link ChangeSetContext} to be cancelled. When the ChangeSet is finished, transactions will be rolled back.
	 */
	void markForCancel();

	/**
	 * @return true, if {@link #markForCancel()} was called at least once, false otherwise
	 */
	boolean isMarkedForCancel();

	/**
	 * Registers a {@link ChangeSetListener} on the {@link ChangeSetContext}, to be notified once the {@link ChangeSetContext} is closed successfully or cancelled.
	 * @param listener the {@link ChangeSetListener}
	 */
	void register(ChangeSetListener listener);
}
